package weather

import (
	"sort"
	"time"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
)

type RequestParams struct {
	geoID     int
	startDate time.Time
	days      int
}

type RequestExtractor struct {
	spanHelper      *span.Helper
	pointComparator *point.Comparator
	clock           clockwork.Clock
}

func NewRequestExtractor(spanHelper *span.Helper, pointComparator *point.Comparator, clock clockwork.Clock) *RequestExtractor {
	return &RequestExtractor{
		spanHelper:      spanHelper,
		pointComparator: pointComparator,
		clock:           clock,
	}
}

type visit struct {
	models.Visit
	inPoint bool // false если начало транспортного спана. То есть пользователь уезжает
}

func (e *RequestExtractor) Extract(trip *tripsmodels.Trip) *RequestParams {
	if trip == nil {
		return nil
	}

	visits := e.extractVisits(trip.GetActiveSpans())
	if len(visits) == 0 {
		return nil
	}

	nextActiveSpanInPoint := e.getNextActiveSpanInPoint(visits)
	if nextActiveSpanInPoint != nil {
		return e.extractRequestFromSpan(nextActiveSpanInPoint)
	}

	// если не нашли спан, то берем точку прибытия
	lastInPointVisit := e.getLastInPointVisit(visits)
	return e.extractRequestFromLastVisit(lastInPointVisit)
}

func (e *RequestExtractor) getNextActiveSpanInPoint(visits []visit) models.Span {
	now := e.clock.Now()
	for i := 0; i < len(visits)-1; i++ {
		left, right := visits[i], visits[i+1]
		if right.When().Before(now) {
			continue
		}
		if !left.inPoint {
			continue
		}
		return models.NewSpan(left.Visit, right.Visit, false)
	}
	return nil
}

func (e *RequestExtractor) extractVisits(spans []models.Span) (visits []visit) {
	for _, s := range spans {
		visits = append(visits, visit{
			Visit:   s.Start(),
			inPoint: !s.IsTransport(),
		})
		visits = append(visits, visit{
			Visit:   s.End(),
			inPoint: true,
		})
	}
	sort.Slice(visits, func(i, j int) bool {
		l, r := visits[i], visits[j]
		if l.When().Equal(r.When()) {
			return l.inPoint && !r.inPoint
		}
		return l.When().Before(r.When())
	})
	return e.deduplicateInPointVisits(visits)
}

func (e *RequestExtractor) deduplicateInPointVisits(visits []visit) (result []visit) {
	if len(visits) == 0 {
		return nil
	}
	result = append(result, visits[0])
	for i, v := range visits[1:] {
		last := result[len(result)-1]
		isLastVisit := i == len(visits)-2
		if e.pointComparator.SamePoints(last.Point(), v.Point()) && v.inPoint && !isLastVisit {
			continue
		}
		result = append(result, v)
	}
	return result
}

func (e *RequestExtractor) getLastInPointVisit(visits []visit) *visit {
	now := e.clock.Now()
	for i := len(visits) - 1; i >= 0; i-- {
		v := visits[i]
		if v.When().Before(now) {
			break
		}
		if v.inPoint {
			return &v
		}
	}
	return nil
}

func (e *RequestExtractor) extractRequestFromSpan(s models.Span) *RequestParams {
	geoID := s.Start().Point().GetGeoID()
	startDate := s.Start().When()
	if startDate.Before(e.clock.Now()) {
		startDate = e.clock.Now()
	}
	endDate := s.End().When()
	if startDate.After(endDate) {
		return nil
	}

	days := getDays(startDate, endDate)
	return &RequestParams{
		geoID:     geoID,
		startDate: helpers.ToDate(startDate),
		days:      days,
	}
}

func (e *RequestExtractor) extractRequestFromLastVisit(v *visit) *RequestParams {
	if v == nil {
		return nil
	}
	return &RequestParams{
		geoID:     v.Point().GetGeoID(),
		startDate: helpers.ToDate(v.When()),
		days:      1,
	}
}

func getDays(left, right time.Time) int {
	if right.Before(left) {
		return 0
	}
	left = helpers.ToDate(left)
	right = helpers.ToDate(right)
	return int(right.Sub(left)/units.Day) + 1
}
