package collector

import (
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/komod/trips/internal/common/dynamicresources"
	"a.yandex-team.ru/travel/komod/trips/internal/common/profiling"
	"a.yandex-team.ru/travel/komod/trips/internal/common/tvm"
	"a.yandex-team.ru/travel/komod/trips/internal/db"
	lbconfig "a.yandex-team.ru/travel/komod/trips/internal/logbroker"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/komod/trips/internal/services/processor"
	"a.yandex-team.ru/travel/komod/trips/internal/services/unprocessedorders"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	logbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
)

type Config struct {
	EnvType          string `config:"YENV_TYPE,required" yaml:"yenv_type"`
	Dicts            references.Config
	DynamicResources dynamicresources.Config `yaml:"dynamic_resources"`
	Logging          logging.Config
	Grpc             grpcserver.GrpcConfig
	HTTP             httpserver.HTTPConfig
	Tvm              tvm.Config
	Logbroker        lbconfig.LogbrokerConfig
	Probes           probes.HTTPConfig
	Profiling        profiling.Config

	Processor         processor.Config
	UnprocessedOrders unprocessedorders.Config `yaml:"unprocessed_orders"`

	OrdersCollectorEnabled                  bool                 `config:"ORDERS_COLLECTOR_ENABLED" yaml:"orders_collector_enabled"`
	OrdersCollectorRPSLimit                 float32              `config:"ORDERS_COLLECTOR_RPS_LIMIT" yaml:"orders_collector_rps_limit"`
	OrdersCollectorInflightLimit            uint                 `config:"ORDERS_COLLECTOR_INFLIGHT_LIMIT" yaml:"orders_collector_inflight_limit"`
	OrdersCollector                         QueueCollectorConfig `yaml:"orders_collector"`
	UnprocessedOrdersCollectorEnabled       bool                 `config:"UNPROCESSED_ORDERS_COLLECTOR_ENABLED" yaml:"unprocessed_orders_collector_enabled"`
	UnprocessedOrdersCollectorRPSLimit      float32              `config:"UNPROCESSED_ORDERS_COLLECTOR_RPS_LIMIT" yaml:"unprocessed_orders_collector_rps_limit"`
	UnprocessedOrdersCollectorInflightLimit uint                 `config:"UNPROCESSED_ORDERS_COLLECTOR_INFLIGHT_LIMIT" yaml:"unprocessed_orders_collector_inflight_limit"`
	UnprocessedOrdersCollector              QueueCollectorConfig `yaml:"unprocessed_orders_collector"`
	OldOrdersCollectorEnabled               bool                 `config:"OLD_ORDERS_COLLECTOR_ENABLED" yaml:"old_orders_collector_enabled"`
	OldOrdersCollectorRPSLimit              float32              `config:"OLD_ORDERS_COLLECTOR_RPS_LIMIT" yaml:"old_orders_collector_rps_limit"`
	OldOrdersCollectorInflightLimit         uint                 `config:"OLD_ORDERS_COLLECTOR_INFLIGHT_LIMIT" yaml:"old_orders_collector_inflight_limit"`
	OldOrdersCollectorMaxReadMessages       uint32               `config:"OLD_ORDERS_COLLECTOR_MAX_READ_MESSAGES"`
	OldOrdersCollector                      QueueCollectorConfig `yaml:"old_orders_collector"`
	Database                                db.Config
}

type QueueCollectorConfig struct {
	Consumer logbroker.ConsumerConfig `yaml:"consumer"`
}

var (
	Cfg = Config{
		EnvType:          "development",
		Dicts:            references.DefaultConfig,
		DynamicResources: dynamicresources.DefaultConfig,
		Logging:          logging.DefaultConfig,
		Grpc:             grpcserver.DefaultGrpcConfig,
		HTTP:             httpserver.DefaultHTTPConfig,
		Tvm:              tvm.DefaultConfig,
		Logbroker:        lbconfig.DefaultConfig,
		Probes:           probes.DefaultHTTPConfig,
		Profiling:        profiling.DefaultConfig,
		Database:         db.DefaultConfig,

		Processor:                  processor.DefaultConfig,
		UnprocessedOrders:          unprocessedorders.DefaultConfig,
		OrdersCollector:            OrdersCollectorCfg,
		UnprocessedOrdersCollector: UnprocessedOrdersCollectorCfg,
		OldOrdersCollector:         OldOrdersCollectorCfg,
	}

	OrdersCollectorCfg = QueueCollectorConfig{
		Consumer: logbroker.ConsumerConfig{
			Topic: logbroker.TopicConfig{
				Path:                 "/travel/test/orders-updates-queue",
				Consumer:             "/avia/development/trips/orders-consumer",
				MaxReadSize:          1024 * 1024,
				MaxReadMessagesCount: 100,
			},
			ReadEndpoints:         lbconfig.AllLogbrokerEndpoints,
			MinActiveConsumers:    2,
			Name:                  "Orders",
			DecompressionDisabled: true,
			Multiline:             false,
		},
	}
	UnprocessedOrdersCollectorCfg = QueueCollectorConfig{
		Consumer: logbroker.ConsumerConfig{
			Topic: logbroker.TopicConfig{
				Path:                 "/avia/development/trips/unprocessed-orders",
				Consumer:             "/avia/development/trips/orders-consumer",
				MaxReadSize:          1024 * 1024,
				MaxReadMessagesCount: 100,
			},
			ReadEndpoints:         lbconfig.AllLogbrokerEndpoints,
			MinActiveConsumers:    2,
			Name:                  "UnprocessedOrders",
			DecompressionDisabled: true,
			Multiline:             false,
		},
	}
	OldOrdersCollectorCfg = QueueCollectorConfig{
		Consumer: logbroker.ConsumerConfig{
			Topic: logbroker.TopicConfig{
				Path:                 "/avia/testing/trips/old-orders",
				Consumer:             "/avia/testing/trips/orders-consumer",
				MaxReadSize:          1024 * 1024,
				MaxReadMessagesCount: 100,
			},
			ReadEndpoints:         lbconfig.AllLogbrokerEndpoints,
			MinActiveConsumers:    2,
			Name:                  "OldOrders",
			DecompressionDisabled: true,
			Multiline:             false,
		},
	}
)
