package dispatchers

import (
	"context"
	"time"

	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/komod/trips/api/processor/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/services"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type base struct {
	logger                   log.Logger
	processorService         services.Processor
	unprocessedEntityService services.UnprocessedOrders
}

func newBase(
	logger log.Logger,
	processorService services.Processor,
	unprocessedEntityService services.UnprocessedOrders,
) *base {
	return &base{
		logger:                   logger,
		processorService:         processorService,
		unprocessedEntityService: unprocessedEntityService,
	}
}

func (b base) dispatchOrder(orderID string, collectedAt *timestamppb.Timestamp, retriesLeft uint32) {
	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	if orderID == "" {
		b.logger.Error("unable to dispatch empty orderID")
		return
	}

	if collectedAt == nil {
		collectedAt = timestamppb.Now()
	}
	req := b.makeOrderEntityRequest(orderID, collectedAt, retriesLeft)

	err := b.processorService.Process(ctx, req)
	if err == nil {
		b.logger.Info("order processed successfully", log.String("orderID", orderID))
		return
	}
	b.logger.Error(
		"unable to process order entity",
		log.Error(err),
		log.String("orderID", orderID),
	)
	if req.RetriesLeft > 0 {
		newReq := proto.Clone(req).(*processor.ProcessTripEntityReq)
		newReq.RetriesLeft--
		b.putAsUnprocessed(newReq)
	} else {
		b.storeFailedRequest(req)
	}
}

func (b *base) makeOrderEntityRequest(
	orderID string,
	collectedAt *timestamppb.Timestamp,
	retriesLeft uint32,
) *processor.ProcessTripEntityReq {
	return &processor.ProcessTripEntityReq{
		CollectedAt: collectedAt,
		Entity: &processor.ProcessTripEntityReq_Order{
			Order: &processor.OrderEntity{
				Id: orderID,
			},
		},
		RetriesLeft: retriesLeft,
	}
}

func (b base) putAsUnprocessed(req *processor.ProcessTripEntityReq) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	if err := b.unprocessedEntityService.Put(ctx, req); err != nil {
		b.logger.Error("unable to store unprocessed message", log.Error(err))
	}
}

func (b base) storeFailedRequest(req *processor.ProcessTripEntityReq) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	err := b.unprocessedEntityService.StoreFailed(ctx, req)
	if err != nil {
		b.logger.Error("failed to store unprocessed order to database", log.String("orderID", req.GetOrder().GetId()))
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, failedToStore).Inc()
	} else {
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, stored).Inc()
	}
}
