package eventslogger

import (
	"time"

	timeformats "cuelang.org/go/pkg/time"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	uaclient "a.yandex-team.ru/library/go/yandex/uagent/log/zap/client"
	eventlogproto "a.yandex-team.ru/travel/komod/trips/internal/components/processor/eventslogger/proto"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/yt/go/yson"
)

type EventType string

func (e EventType) String() string {
	return string(e)
}

const (
	EventTypeCreateTrip = "CREATE_TRIP"
	EventTypeUpdateTrip = "UPDATE_TRIP"
	EventTypeDeleteTrip = "DELETE_TRIP"

	logName = "processing-events-log"
)

type EventsLogger struct {
	logger   log.Logger
	uaClient uaclient.Client
}

func NewEventsLogger(logger log.Logger, uaClient uaclient.Client) *EventsLogger {
	return &EventsLogger{logger: logger, uaClient: uaClient}
}

func (el *EventsLogger) Log(eventType EventType, trip *tripsmodels.Trip, orderID orders.ID) {
	startPoint, endPoint, startDate, endDate := "", "", "", ""
	if len(trip.OrderInfos) != 0 {
		startPoint = point.ExtractPointKey(trip.StartVisit().Point())
		endPoint = point.ExtractPointKey(trip.EndVisit().Point())
		startDate = trip.StartTime().Format(timeformats.RFC3339)
		endDate = trip.EndTime().Format(timeformats.RFC3339)
	}
	record := &eventlogproto.ProcessingEvent{
		Timestamp:  time.Now().Unix(),
		PassportID: trip.PassportID,
		TripID:     trip.ID,
		OrderID:    orderID.String(),
		EventType:  eventType.String(),
		StartPoint: startPoint,
		EndPoint:   endPoint,
		StartDate:  startDate,
		EndDate:    endDate,
		Spans:      el.getSpansPayload(trip),
		OrderIds:   el.getOrderIDs(trip),
	}
	payload, err := proto.Marshal(record)
	if err != nil {
		el.logger.Error("failed to marshal processing-events-log record", log.Error(err))
	}
	err = el.uaClient.Send([]uaclient.Message{{
		Payload: payload,
		Meta:    map[string]string{"log_name": logName},
		Time:    ptr.Time(time.Now().UTC()),
	}})
	if err != nil {
		el.logger.Error("failed to send processing-events-log record to UnifiedAgent", log.Error(err))
	}
}

func (el *EventsLogger) getSpansPayload(trip *tripsmodels.Trip) []byte {
	spans := make([]marshalableSpan, 0)
	for _, s := range trip.GetSpans() {
		spans = append(spans, newMarshalableSpan(s))
	}
	spansPayload, err := yson.Marshal(spans)
	if err != nil {
		el.logger.Error("failed to marshal spans for processing-events-log record")
	}
	return spansPayload
}

func (el *EventsLogger) getOrderIDs(trip *tripsmodels.Trip) []string {
	orderIDs := make([]string, 0, len(trip.OrderInfos))
	for _, orderID := range trip.GetOrderIDs() {
		orderIDs = append(orderIDs, string(orderID))
	}
	return orderIDs
}
