package processor

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/gofrs/uuid"
	opentracing "github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/komod/trips/internal/components/processor/eventslogger"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/trips"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/errutil"
)

type orderInfoExtractor interface {
	Extract(order orders.Order) (tripsmodels.OrderInfo, error)
}

type eventsLogger interface {
	Log(eventslogger.EventType, *tripsmodels.Trip, orders.ID)
}

type Processor struct {
	logger             log.Logger
	ordersClient       orders.Client
	orderInfoExtractor orderInfoExtractor
	tripsMatcher       trips.Matcher
	tripsStorage       trips.Storage
	eventsLogger       eventsLogger
	metrics            Metrics
}

func NewProcessor(
	logger log.Logger,
	ordersClient orders.Client,
	orderInfoExtractor orderInfoExtractor,
	tripsMerger trips.Matcher,
	tripsStorage trips.Storage,
	eventsLogger eventsLogger,
) *Processor {
	return &Processor{
		logger:             logger.WithName("OrderProcessor"),
		ordersClient:       ordersClient,
		orderInfoExtractor: orderInfoExtractor,
		tripsMatcher:       tripsMerger,
		tripsStorage:       tripsStorage,
		eventsLogger:       eventsLogger,
		metrics:            NewMetrics(),
	}
}

func (p *Processor) ProcessOrder(ctx context.Context, orderID orders.ID, collectedAt time.Time) (err error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "Processor.ProcessOrder", opentracing.Tag{Key: "orderID", Value: orderID})
	defer tracingSpan.Finish()
	defer errutil.Wrap(&err, "processor.Processor.ProcessOrder")

	ctx = ctxlog.WithFields(ctx, log.String("orderID", orderID.String()))
	ctxlog.Info(ctx, p.logger, "start processing order")
	defer ctxlog.Info(ctx, p.logger, "finished processing order")

	order, err := p.ordersClient.GetOrderNoAuth(ctx, orderID)
	if err != nil {
		if errors.Is(err, orders.ErrOrderValidation{}) {
			p.logger.Warn("skip invalid order", log.Error(err))
			return nil
		}
		return err
	}

	if p.needSkipOrder(ctx, order) {
		return nil
	}
	ctx = ctxlog.WithFields(ctx, log.String("passportID", order.PassportID()))
	orderInfo, err := p.orderInfoExtractor.Extract(order)
	if err != nil {
		p.logger.Error("failed to extract order info", log.Error(err), log.String("orderID", order.ID().String()))
		return err
	}

	var isNewOrder bool
	var tripByOrder *tripsmodels.Trip
	err = p.tripsStorage.ExecuteInTransaction(
		func(tx trips.StorageTxSession) error {
			if err := tx.LockUser(ctx, order.PassportID()); err != nil {
				return err
			}

			userTrips, err := tx.GetTrips(ctx, order.PassportID())
			if err != nil {
				return err
			}

			tripByOrder = findTripByOrderID(order.ID(), userTrips)
			if tripByOrder == nil {
				if order.Cancelled() {
					return nil
				}
				isNewOrder = true
				return p.updateTripsWithNewOrder(ctx, tx, orderInfo, order.PassportID(), userTrips)
			} else {
				return p.updateExistingOrder(ctx, tx, order, orderInfo, tripByOrder)
			}
		},
	)
	if err != nil {
		p.logger.Error("failed to process order", log.String("orderID", orderID.String()), log.Error(err))
	} else {
		p.logger.Info("successfully processed order", log.String("orderID", orderID.String()))
	}

	if isNewOrder && err == nil {
		p.registerOrderDeliveryLag(collectedAt)
	}

	return err
}

func (p *Processor) needSkipOrder(ctx context.Context, order orders.Order) bool {
	if order == nil {
		ctxlog.Info(ctx, p.logger, "skip order of unsupported type")
		return true
	}
	if order.InProgress() {
		ctxlog.Info(ctx, p.logger, "skip processing for in-progress order")
		return true
	}
	if order.PassportID() == "" {
		ctxlog.Info(ctx, p.logger, "skip processing for unknown user")
		return true
	}
	if !Cfg.ProcessBusOrders {
		ctxlog.Info(ctx, p.logger, "skip processing for bus order")
		return true
	}
	return false
}

func (p *Processor) updateTripsWithNewOrder(
	ctx context.Context,
	tx trips.StorageTxSession,
	orderInfo tripsmodels.OrderInfo,
	passportID string,
	currentTrips tripsmodels.Trips,
) (err error) {
	defer errutil.Wrap(&err, "processor.Processor.updateTripsWithNewOrder")

	matchedTrips := p.tripsMatcher.MatchTripsWithConnectedSpans(currentTrips, orderInfo.Spans...)

	combinedTrip := combineTrips(orderInfo, passportID, matchedTrips...)
	if err = tx.UpsertTrips(ctx, combinedTrip); err != nil {
		return err
	}
	p.eventsLogger.Log(eventslogger.EventTypeCreateTrip, combinedTrip, orderInfo.ID)
	if err = tx.RemoveTrips(ctx, matchedTrips...); err != nil {
		return err
	}
	for _, trip := range matchedTrips {
		p.eventsLogger.Log(eventslogger.EventTypeDeleteTrip, trip, orderInfo.ID)
	}
	return nil
}

func (p *Processor) updateExistingOrder(
	ctx context.Context,
	tx trips.StorageTxSession,
	order orders.Order,
	orderInfo tripsmodels.OrderInfo,
	trip *tripsmodels.Trip,
) error {
	const funcName = "processor.Processor.updateExistingOrder"

	if order.Cancelled() {
		return p.handleExistingOrderCancelled(ctx, tx, order, trip)
	}
	trip.UpsertOrder(orderInfo)
	err := tx.UpsertTrips(ctx, trip)
	if err != nil {
		return fmt.Errorf("%s: %w", funcName, err)
	}
	p.eventsLogger.Log(eventslogger.EventTypeUpdateTrip, trip, orderInfo.ID)
	return nil
}

func (p *Processor) handleExistingOrderCancelled(
	ctx context.Context,
	tx trips.StorageTxSession,
	order orders.Order,
	trip *tripsmodels.Trip,
) error {
	delete(trip.OrderInfos, order.ID())
	if err := tx.RemoveTripOrderSpans(ctx, trip.ID); err != nil {
		return fmt.Errorf("failed to remove order spans for trip: %w", err)
	}
	if len(trip.OrderInfos) == 0 {
		if err := tx.RemoveTrips(ctx, trip); err != nil {
			return fmt.Errorf("failed to remove trip: %w", tx.RemoveTrips(ctx, trip))
		}
		ctxlog.Info(ctx, p.logger, "removed trip with only cancelled order", log.String("tripID", trip.ID))
		p.eventsLogger.Log(eventslogger.EventTypeDeleteTrip, trip, order.ID())
		return nil
	}
	if err := tx.UpsertTrips(ctx, trip); err != nil {
		return fmt.Errorf("failed to upsert trip: %w", err)
	}
	p.eventsLogger.Log(eventslogger.EventTypeUpdateTrip, trip, order.ID())
	return nil
}

func (p *Processor) registerOrderDeliveryLag(collectedAt time.Time) {
	p.metrics.orderDeliveryLag.RecordDuration(time.Since(collectedAt))
}

func combineTrips(orderInfo tripsmodels.OrderInfo, passportID string, matchedTrips ...*tripsmodels.Trip) *tripsmodels.Trip {
	newTrip := tripsmodels.NewTrip(uuid.Must(uuid.NewV4()).String(), passportID)

	newTrip.UpsertOrder(orderInfo)
	for _, trip := range matchedTrips {
		for _, tripOrderInfo := range trip.OrderInfos {
			newTrip.UpsertOrder(tripOrderInfo)
		}
	}
	return newTrip
}

func findTripByOrderID(orderID orders.ID, currentTrips tripsmodels.Trips) *tripsmodels.Trip {
	for _, trip := range currentTrips {
		if _, found := trip.OrderInfos[orderID]; found {
			return trip
		}
	}
	return nil
}
