package db

import (
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/trips/matcher"
	tripmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	ordercommons "a.yandex-team.ru/travel/orders/proto"
)

type ObjectsModelsMapper struct {
	spanVisitFactory *point.Factory
}

func (m ObjectsModelsMapper) MapTripObjectsToTripModels(items ...*tripmodels.Trip) []Trip {
	trips := make([]Trip, 0, len(items))
	for _, trip := range items {
		trips = append(trips, Trip{
			ID:         trip.ID,
			PassportID: trip.PassportID,
		})
	}
	return trips
}

func (m ObjectsModelsMapper) MapTripObjectsToOrderSpanModels(items ...*tripmodels.Trip) []OrderSpan {
	spans := make([]OrderSpan, 0)
	for _, trip := range items {
		for orderID, orderInfo := range trip.OrderInfos {
			for _, orderSpan := range orderInfo.Spans {
				newSpan := OrderSpan{
					PassportID:        trip.PassportID,
					TripID:            trip.ID,
					OrderID:           orderID.String(),
					StartPoint:        point.ExtractPointKey(orderSpan.Start().Point()),
					EndPoint:          point.ExtractPointKey(orderSpan.End().Point()),
					StartTime:         orderSpan.Start().When(),
					EndTime:           orderSpan.End().When(),
					IsTransport:       orderSpan.IsTransport(),
					MatcherVersion:    matcher.Version,
					DisplayOrderState: uint8(orderInfo.State),
				}
				spans = append(
					spans,
					newSpan,
				)
			}
		}
	}
	return spans
}

func (m ObjectsModelsMapper) MapOrderSpanModelsToTripObjects(orderSpanModels []OrderSpan) (tripmodels.Trips, error) {
	tripsMap := make(map[string]*tripmodels.Trip)
	for _, dbOrderSpan := range orderSpanModels {
		trip := tripsMap[dbOrderSpan.TripID]
		if trip == nil {
			trip = tripmodels.NewTrip(dbOrderSpan.TripID, dbOrderSpan.PassportID)
			tripsMap[dbOrderSpan.TripID] = trip
		}

		orderID := orders.ID(dbOrderSpan.OrderID)
		orderInfo, found := trip.OrderInfos[orderID]
		if !found {
			state := ordercommons.EDisplayOrderState(dbOrderSpan.DisplayOrderState)
			if state == 0 {
				if dbOrderSpan.Cancelled {
					state = ordercommons.EDisplayOrderState_OS_REFUNDED
				} else {
					state = ordercommons.EDisplayOrderState_OS_FULFILLED
				}
			}
			orderInfo = tripmodels.OrderInfo{
				ID:    orderID,
				State: state,
				Spans: make([]models.Span, 0),
			}
		}

		newSpan, err := m.buildSpanModel(dbOrderSpan)
		if err != nil {
			return nil, err
		}
		orderInfo.Spans = append(
			orderInfo.Spans,
			newSpan,
		)
		trip.OrderInfos[orderID] = orderInfo
	}
	result := make(tripmodels.Trips, 0, len(tripsMap))
	for _, trip := range tripsMap {
		result = append(result, trip)
	}
	return result, nil

}

func (m ObjectsModelsMapper) buildSpanModel(orderSpan OrderSpan) (models.Span, error) {
	startPoint, err := m.spanVisitFactory.MakeByPointKey(orderSpan.StartPoint)
	if err != nil {
		return nil, err
	}
	endPoint, err := m.spanVisitFactory.MakeByPointKey(orderSpan.EndPoint)
	if err != nil {
		return nil, err
	}
	return newMappedSpan(
		orderSpan.ID,
		models.NewVisit(startPoint, orderSpan.StartTime),
		models.NewVisit(endPoint, orderSpan.EndTime),
		orderSpan.IsTransport,
	), nil
}
