package db

import (
	"context"
	"fmt"
	"strings"

	opentracing "github.com/opentracing/opentracing-go"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/trips/models"
)

type BadSession struct{}

func (b BadSession) GetTrips(_ context.Context, _ string) (models.Trips, error) {
	return nil, pgclient.ErrNodeIsUnavailable
}

func (b BadSession) GetTrip(_ context.Context, _ string) (*models.Trip, error) {
	return nil, pgclient.ErrNodeIsUnavailable
}

func (b BadSession) LockUser(_ context.Context, _ string) error {
	return pgclient.ErrNodeIsUnavailable
}

func (b BadSession) UpsertTrips(_ context.Context, _ ...*models.Trip) error {
	return pgclient.ErrNodeIsUnavailable
}

func (b BadSession) RemoveTrips(_ context.Context, _ ...*models.Trip) error {
	return pgclient.ErrNodeIsUnavailable
}

func (b BadSession) RemoveTripOrderSpans(_ context.Context, _ string) error {
	return pgclient.ErrNodeIsUnavailable
}

type Session struct {
	db      *gorm.DB
	storage *TripsStorage
}

func (s *Session) GetTrips(ctx context.Context, passportID string) (models.Trips, error) {
	const funcName = "trips.db.Session.GetTrips"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()

	spans := make([]OrderSpan, 0)

	query := s.getDB(ctx).Where("passport_id = ?", passportID)
	if err := query.Find(&spans).Error; err != nil {
		return nil, err
	}

	result, err := s.storage.objectsModelsMapper.MapOrderSpanModelsToTripObjects(spans)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}
	return result, nil
}

func (s *Session) GetTrip(ctx context.Context, tripID string) (*models.Trip, error) {
	const funcName = "trips.db.Session.GetTrip"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()

	spans := make([]OrderSpan, 0)

	query := s.getDB(ctx).
		Where("trip_id = ?", tripID)
	if err := query.Find(&spans).Error; err != nil {
		return nil, err
	}

	if len(spans) == 0 {
		return nil, nil
	}

	result, err := s.storage.objectsModelsMapper.MapOrderSpanModelsToTripObjects(spans)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}
	if len(result) != 1 {
		return nil, fmt.Errorf("unexpected trip data tripID=%s", tripID)
	}

	return result[0], nil
}

func (s Session) getDB(ctx context.Context) *gorm.DB {
	return s.db.WithContext(ctx)
}

type TxSession struct {
	Session
	txOptions TransactionOptions
}

func (s *TxSession) LockUser(ctx context.Context, passportID string) error {
	const funcName = "trips.db.TxSession.LockUser"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()

	if err := s.setupServerTxOptions(ctx); err != nil {
		return err
	}

	lockingOptions := "NOWAIT"
	if s.txOptions.WaitForLock {
		lockingOptions = ""
	}

	err := s.getDB(ctx).
		Clauses(clause.OnConflict{DoNothing: true}).
		Clauses(clause.Locking{Strength: "UPDATE", Options: lockingOptions}).
		Create(&User{PassportID: passportID}).
		Error

	if err != nil {
		if strings.Contains(err.Error(), "canceling statement due to lock timeout") {
			return ErrUnableToLockUser
		}
		return err
	}

	user := User{}
	err = s.getDB(ctx).
		Clauses(clause.Locking{Strength: "UPDATE", Options: lockingOptions}).
		Find(&user, "passport_id = ?", passportID).
		Error

	if err != nil && strings.Contains(err.Error(), "could not obtain lock on row in relation") {
		return ErrUnableToLockUser
	}

	return err
}

func (s *TxSession) UpsertTrips(ctx context.Context, items ...*models.Trip) error {
	const funcName = "trips.db.TxSession.UpsertTrips"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()

	tripModels := s.storage.objectsModelsMapper.MapTripObjectsToTripModels(items...)
	orderSpanModels := s.storage.objectsModelsMapper.MapTripObjectsToOrderSpanModels(items...)

	db := s.getDB(ctx)
	db.Clauses(clause.OnConflict{DoNothing: true}).Create(&tripModels)

	orderIDs := s.collectOrderIDs(orderSpanModels)
	if err := db.Where("order_id in ?", orderIDs).Delete(OrderSpan{}).Error; err != nil {
		return err
	}

	return db.Save(orderSpanModels).Error
}

func (s *TxSession) RemoveTripOrderSpans(ctx context.Context, tripID string) error {
	const funcName = "trips.db.TxSession.RemoveTripOrderSpans"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()
	return s.getDB(ctx).Delete(OrderSpan{}, OrderSpan{TripID: tripID}).Error
}

func (s *TxSession) RemoveTrips(ctx context.Context, items ...*models.Trip) error {
	const funcName = "trips.db.TxSession.RemoveTrips"
	tracedSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracedSpan.Finish()

	tripIDs := make([]string, 0)
	for _, item := range items {
		if item.ID != "" {
			tripIDs = append(tripIDs, item.ID)
		}
	}
	if len(tripIDs) == 0 {
		return nil
	}
	return s.getDB(ctx).Delete(Trip{}, tripIDs).Error
}

func (s *TxSession) collectOrderIDs(orderSpanModels []OrderSpan) []orders.ID {
	orderIDsMap := make(map[orders.ID]struct{})
	for _, model := range orderSpanModels {
		orderIDsMap[orders.ID(model.OrderID)] = struct{}{}
	}
	orderIDs := make([]orders.ID, 0)
	for id := range orderIDsMap {
		orderIDs = append(orderIDs, id)
	}
	return orderIDs
}

func (s *TxSession) setupServerTxOptions(ctx context.Context) error {
	err := s.getDB(ctx).
		Exec(
			fmt.Sprintf(
				"SET LOCAL statement_timeout = %d;",
				int(s.txOptions.StatementTimeout.Milliseconds()),
			),
		).Error
	if err != nil {
		return err
	}

	err = s.getDB(ctx).
		Exec(
			fmt.Sprintf(
				"SET LOCAL idle_in_transaction_session_timeout = %d;",
				int(s.txOptions.IdleInTransactionSessionTimeout.Milliseconds()),
			),
		).Error
	if err != nil {
		return err
	}

	lockTimeout := 1
	if s.txOptions.WaitForLock {
		lockTimeout = int(s.txOptions.LockTimeout.Milliseconds())
	}
	err = s.getDB(ctx).Exec(
		fmt.Sprintf(
			"SET LOCAL lock_timeout = %d;",
			lockTimeout,
		),
	).Error
	if err != nil {
		return err
	}
	return nil
}
