package extractors

import (
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type pointFactory interface {
	MakeByStationID(stationID int) (models.Point, error)
	MakeBySettlementID(settlementID int) (models.Point, error)
}

type OrderInfoExtractor struct {
	spanPointFactory pointFactory
}

func NewOrderInfoExtractor(spanPointFactory pointFactory) *OrderInfoExtractor {
	return &OrderInfoExtractor{spanPointFactory: spanPointFactory}
}

func (e OrderInfoExtractor) Extract(order orders.Order) (_ tripsmodels.OrderInfo, err error) {
	var spans []models.Span
	switch value := order.(type) {
	case *orders.HotelOrder:
		spans = e.extractHotelSpans(value)
	case *orders.TrainOrder:
		spans, err = e.extractRouteSpans(value.Route)
	case *orders.BusOrder:
		spans, err = e.extractRouteSpans(value.Route)
	case *orders.AviaOrder:
		spans, err = e.extractRouteSpans(value.Route)
	}
	return tripsmodels.NewOrderInfo(order.ID(), order.State(), spans), err
}

func (e OrderInfoExtractor) extractHotelSpans(order *orders.HotelOrder) []models.Span {
	return []models.Span{models.NewSpan(
		models.NewVisit(order.Point, order.CheckinDate),
		models.NewVisit(order.Point, order.CheckoutDate),
		false,
	)}
}

func (e OrderInfoExtractor) extractPointFromStationSettlement(station *rasp.TStation, settlement *rasp.TSettlement) (models.Point, error) {
	if settlement != nil {
		return e.spanPointFactory.MakeBySettlementID(int(settlement.Id))
	} else {
		return e.spanPointFactory.MakeByStationID(int(station.Id))
	}
}

func (e OrderInfoExtractor) extractRouteSpans(route orders.Route) (result []models.Span, err error) {
	from, err := e.extractPointFromStationSettlement(route.FromStation, route.FromSettlement)
	if err != nil {
		return nil, err
	}
	to, err := e.extractPointFromStationSettlement(route.ToStation, route.ToSettlement)
	if err != nil {
		return nil, err
	}
	result = append(result, models.NewSpan(
		models.NewVisit(from, route.ForwardDeparture),
		models.NewVisit(to, route.ForwardArrival),
		true,
	))

	if route.BackwardDeparture != nil {
		result = append(result, models.NewSpan(
			models.NewVisit(to, *route.BackwardDeparture),
			models.NewVisit(from, *route.BackwardArrival),
			true,
		))
	}
	return result, nil
}
