package http

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"

	tvmutil "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

type Fetcher struct {
	backOffPolicyGetter func() backoff.BackOff
	httpClient          *http.Client
	tvmClient           tvm.Client
	tvmID               uint32
	requestTimeout      time.Duration
}

func NewHTTPFetcher(tvmID uint32, httpClient *http.Client, tvmClient tvm.Client, opts ...Option) *Fetcher {
	c := &Fetcher{
		tvmID:      tvmID,
		httpClient: httpClient,
		tvmClient:  tvmClient,
	}
	defaultOpts := []Option{withDefaultBackOffPolicy, withDefaultRequestTimeout}
	for _, opt := range append(defaultOpts, opts...) {
		opt(c)
	}
	return c
}

func (c *Fetcher) Get(request *http.Request) ([]byte, error) {
	var byteResponse []byte
	requestFunc := func() error {
		response, err := c.httpClient.Do(request)
		if err != nil {
			return RequestErr{err}
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return RequestErr{err}
		}
		if response.StatusCode == http.StatusNotFound {
			return backoff.Permanent(NewNotFoundErr(string(byteResponse)))
		}
		if response.StatusCode != http.StatusOK {
			err := fmt.Errorf("travel api answered with code %d: %s", response.StatusCode, string(byteResponse))
			if response.StatusCode == http.StatusBadRequest {
				return backoff.Permanent(err)
			}
			return RequestErr{err}
		}
		return nil
	}
	if err := c.retryRequest(requestFunc); err != nil {
		return nil, err
	}
	return byteResponse, nil
}

func (c *Fetcher) SetServiceTicket(request *http.Request) error {
	var tvmTicket string
	if c.tvmID != 0 {
		var err error
		tvmTicket, err = c.tvmClient.GetServiceTicketForID(request.Context(), tvm.ClientID(c.tvmID))
		if err != nil {
			return err
		}
	}
	if tvmTicket != "" {
		request.Header.Set(tvmutil.XYaServiceTicket, tvmTicket)
	}
	return nil
}

func (c *Fetcher) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}
