package middlewares

import (
	"net/http"
	"regexp"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors/multierr"
)

type Middleware = func(next http.Handler) http.Handler

type Router struct {
	logger     log.Logger
	middleware Middleware

	buildError       error
	includedPatterns []*regexp.Regexp
	excludedPatterns []*regexp.Regexp
}

func New(logger log.Logger, middleware Middleware) *Router {
	return &Router{logger: logger, middleware: middleware}
}

func (b *Router) Include(patterns ...string) *Router {
	b.includedPatterns = append(b.includedPatterns, b.buildPatterns(patterns)...)
	return b
}

func (b *Router) Exclude(patterns ...string) *Router {
	b.excludedPatterns = append(b.excludedPatterns, b.buildPatterns(patterns)...)
	return b
}

func (b *Router) MustBuild() Middleware {
	mw, err := b.Build()
	if err != nil {
		panic(err)
	}
	return mw
}

func (b *Router) Build() (Middleware, error) {
	return func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			mw := b.Match(r.URL.Path)
			handler := next
			if mw != nil {
				handler = mw(next)
			}
			handler.ServeHTTP(w, r)
		})
	}, b.buildError
}

func (b *Router) Match(endpoint string) Middleware {
	for _, pattern := range b.excludedPatterns {
		if pattern.MatchString(endpoint) {
			return nil
		}
	}
	for _, pattern := range b.includedPatterns {
		if pattern.MatchString(endpoint) {
			return b.middleware
		}
	}
	return nil
}

func (b *Router) buildPatterns(patterns []string) (result []*regexp.Regexp) {
	for _, pattern := range patterns {
		reg, compileErr := regexp.Compile(pattern)
		if reg != nil {
			result = append(result, reg)
		}
		b.buildError = multierr.Append(b.buildError, compileErr)
	}
	return result
}
