package models

import (
	"fmt"
	"strconv"
	"time"
)

type PointKind int

const (
	UnknownPointKind PointKind = iota
	GeoRegionPointKind
	StationPointKind
	SettlementPointKind
	RegionPointKind
	CountryPointKind
)

type Point interface {
	GetTitle() string
	GetTimezone() *time.Location
	GetKind() PointKind
	GetID() int
	GetGeoID() int
	GetLinguistics() Linguistics
	GetPointKey() string
	isPoint()
}

type usingKind struct {
	kind PointKind
}

func (u usingKind) GetKind() PointKind {
	return u.kind
}

type FakePoint struct {
	usingKind
	id int
}

func NewFakePoint(id int) Point {
	return &FakePoint{
		usingKind: usingKind{kind: UnknownPointKind},
		id:        id,
	}
}

func (f FakePoint) GetTitle() string {
	return strconv.Itoa(f.id)
}

func (f FakePoint) GetTimezone() *time.Location {
	return nil
}

func (f FakePoint) GetLinguistics() Linguistics {
	return Linguistics{NominativeCase: f.GetTitle()}
}

func (f FakePoint) GetGeoID() int {
	return f.id
}

func (f FakePoint) GetID() int {
	return f.id
}

func (f FakePoint) GetPointKey() string {
	return fmt.Sprintf("fake%d", f.id)
}

func (f FakePoint) isPoint() {}

type GeoRegionPoint struct {
	usingKind
	geoID       int
	timezone    *time.Location
	linguistics Linguistics
}

func NewGeoRegionPoint(geoID int, timezone *time.Location, linguistics Linguistics) *GeoRegionPoint {
	return &GeoRegionPoint{
		usingKind:   usingKind{kind: GeoRegionPointKind},
		geoID:       geoID,
		timezone:    timezone,
		linguistics: linguistics,
	}
}

func (p GeoRegionPoint) GetGeoID() int {
	return p.geoID
}

func (p GeoRegionPoint) GetID() int {
	return p.geoID
}

func (p GeoRegionPoint) GetTitle() string {
	return p.linguistics[NominativeCase]
}

func (p GeoRegionPoint) GetTimezone() *time.Location {
	return p.timezone
}

func (p GeoRegionPoint) GetLinguistics() Linguistics {
	return p.linguistics
}

func (p GeoRegionPoint) GetPointKey() string {
	return fmt.Sprintf("g%d", p.GetGeoID())
}

func (p GeoRegionPoint) isPoint() {}

type StationPoint struct {
	usingKind
	nearestGeoID int
	timezone     *time.Location
	stationID    int
	linguistics  Linguistics
}

func NewStationPoint(stationID, nearestGeoID int, timezone *time.Location, linguistics Linguistics) *StationPoint {
	return &StationPoint{
		usingKind:    usingKind{kind: StationPointKind},
		nearestGeoID: nearestGeoID,
		timezone:     timezone,
		stationID:    stationID,
		linguistics:  linguistics,
	}
}

func (p StationPoint) GetGeoID() int {
	return p.nearestGeoID
}

func (p StationPoint) GetTimezone() *time.Location {
	return p.timezone
}

func (p StationPoint) GetID() int {
	return p.stationID
}

func (p StationPoint) GetTitle() string {
	return p.linguistics[NominativeCase]
}

func (p StationPoint) GetLinguistics() Linguistics {
	return p.linguistics
}

func (p StationPoint) GetPointKey() string {
	return fmt.Sprintf("s%d", p.GetID())
}

func (p StationPoint) isPoint() {}

type SettlementPoint struct {
	usingKind
	settlementID int
	timezone     *time.Location
	linguistics  Linguistics
}

func NewSettlementPoint(settlementID int, timezone *time.Location, linguistics Linguistics) *SettlementPoint {
	return &SettlementPoint{
		usingKind:    usingKind{kind: SettlementPointKind},
		settlementID: settlementID,
		timezone:     timezone,
		linguistics:  linguistics,
	}
}

func (p SettlementPoint) GetGeoID() int {
	return 0
}

func (p SettlementPoint) GetTimezone() *time.Location {
	return p.timezone
}

func (p SettlementPoint) GetID() int {
	return p.settlementID
}

func (p SettlementPoint) GetTitle() string {
	return p.linguistics[NominativeCase]
}

func (p SettlementPoint) GetLinguistics() Linguistics {
	return p.linguistics
}

func (p SettlementPoint) GetPointKey() string {
	return fmt.Sprintf("c%d", p.GetID())
}

func (p SettlementPoint) isPoint() {}

type CountryPoint struct {
	usingKind
	countryID   int
	linguistics Linguistics
}

func NewCountryPoint(countryID int, linguistics Linguistics) *CountryPoint {
	return &CountryPoint{
		usingKind:   usingKind{kind: CountryPointKind},
		countryID:   countryID,
		linguistics: linguistics,
	}
}

func (p CountryPoint) GetGeoID() int {
	return 0
}

func (p CountryPoint) GetTimezone() *time.Location {
	return nil
}

func (p CountryPoint) GetID() int {
	return p.countryID
}

func (p CountryPoint) GetTitle() string {
	return p.linguistics[NominativeCase]
}

func (p CountryPoint) GetLinguistics() Linguistics {
	return p.linguistics
}

func (p CountryPoint) GetPointKey() string {
	return fmt.Sprintf("l%d", p.GetID())
}

func (p CountryPoint) isPoint() {}

type RegionPoint struct {
	usingKind
	regionID    int
	linguistics Linguistics
}

func NewRegionPoint(regionID int, linguistics Linguistics) *RegionPoint {
	return &RegionPoint{
		usingKind:   usingKind{kind: RegionPointKind},
		regionID:    regionID,
		linguistics: linguistics,
	}
}

func (p RegionPoint) GetGeoID() int {
	return 0
}

func (p RegionPoint) GetTimezone() *time.Location {
	return nil
}

func (p RegionPoint) GetID() int {
	return p.regionID
}

func (p RegionPoint) GetTitle() string {
	return p.linguistics[NominativeCase]
}

func (p RegionPoint) GetLinguistics() Linguistics {
	return p.linguistics
}

func (p RegionPoint) GetPointKey() string {
	return fmt.Sprintf("r%d", p.GetID())
}

func (p RegionPoint) isPoint() {}
