package notifier

import (
	"context"
	"fmt"
	"time"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/balancer"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	hcbalancer "a.yandex-team.ru/travel/library/go/grpcutil/client/healthcheck_balancer"
	"a.yandex-team.ru/travel/library/go/grpcutil/client/ypresolver"
	"a.yandex-team.ru/travel/library/go/tvm"
	pullnotifications "a.yandex-team.ru/travel/notifier/api/pull_notifications/v1"
)

type Client interface {
	GetNotificationsByOrderIDs(ctx context.Context, orderIDs []string) (*pullnotifications.GetPullNotificationsByOrdersIdsRspV1, error)
}

type GrpcNotifierClient struct {
	tvmInterceptor grpc.UnaryClientInterceptor
	logger         log.Logger
	grpcClient     pullnotifications.PullNotificationsAPIClient
	config         Config
}

func initNotifierBalancer(config Config, selfTvmID uint32) {
	notifierHealthCheckServiceName := "notifier-api"
	logger, _ := zap.NewDeployLogger(log.DebugLevel)
	balancer.Register(
		hcbalancer.NewBalancerBuilder(
			hcbalancer.WithLogger(logger.WithName("NotifierBalancer")),
			hcbalancer.WithHealthCheckServiceName(notifierHealthCheckServiceName),
			hcbalancer.WithHealthCheckInterval(2000*time.Millisecond),
			hcbalancer.WithBalancingMethod(hcbalancer.BalancingMethodChooseClosest),
			hcbalancer.WithTVMProvider(
				func() (*tvm.TvmHelper, uint32) {
					tvmHelper := tvm.NewDeployTvmHelper(
						logger,
						&tvm.TvmHelperConfig{
							SelfID:    selfTvmID,
							WhiteList: []uint32{config.TvmID},
						},
					)
					return &tvmHelper, config.TvmID
				},
			),
		),
	)
}

func CreateConnection(selfTvmID uint32, config Config, logger log.Logger) (grpc.ClientConnInterface, error) {
	if !config.Enabled {
		return nil, nil
	}
	initNotifierBalancer(config, selfTvmID)
	ctx := context.Background()
	ctx, cancel := context.WithTimeout(ctx, 90*time.Second)
	defer cancel()
	resolver := ypresolver.NewYPResolverBuilder(ypresolver.WithLogger(logger.Structured()))
	serviceConfig := hcbalancer.ServiceConfig{
		LoadBalancingPolicy: "healthcheck_balancer",
		LoadBalancingConfig: hcbalancer.NewLoadBalancingConfig(
			hcbalancer.NewLoadBalancerConfig("healthcheck_balancer", config.ConnectionsPerHost),
		),
	}
	tvmHelper := tvm.NewDeployTvmHelper(
		logger, &tvm.TvmHelperConfig{
			SelfID:    selfTvmID,
			WhiteList: []uint32{},
		},
	)
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(config.TvmID)
	if tvmInterceptor == nil {
		return nil, fmt.Errorf("failed to create grpc interceptor for notifier client")
	}
	return grpc.DialContext(
		ctx,
		ypresolver.BuildServiceFQDN(config.YPlannerID),
		grpc.WithResolvers(resolver),
		grpc.WithDefaultServiceConfig(serviceConfig.MarshalToJSON()),
		grpc.WithInsecure(),
		grpc.WithBlock(),
		grpc.WithUnaryInterceptor(tvmInterceptor),
	)
}

func NewNotificationsClient(
	connection grpc.ClientConnInterface,
	config Config,
	logger log.Logger,
) *GrpcNotifierClient {
	grpcClient := pullnotifications.NewPullNotificationsAPIClient(connection)
	return &GrpcNotifierClient{
		logger:     logger,
		config:     config,
		grpcClient: grpcClient,
	}
}

func (pc *GrpcNotifierClient) GetNotificationsByOrderIDs(ctx context.Context, orderIDs []string) (*pullnotifications.GetPullNotificationsByOrdersIdsRspV1, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "NotificationsClient.GetNotificationsByOrderIDs")
	defer tracingSpan.Finish()
	ctx, cancel := context.WithTimeout(ctx, pc.config.RequestTimeout)
	defer cancel()

	if !pc.config.Enabled {
		return &pullnotifications.GetPullNotificationsByOrdersIdsRspV1{}, nil
	}
	req := pullnotifications.GetPullNotificationsByOrdersIdsReqV1{OrderIds: orderIDs}
	return pc.grpcClient.GetByOrderIDs(ctx, &req)
}
