package clients

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/opentracing/opentracing-go"

	tvmutil "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/models"
	"a.yandex-team.ru/travel/komod/trips/internal/usercredentials"
)

type Config struct {
	TravelAPIHost         string        `yaml:"travel_api_host"`
	TravelAPITvmID        uint32        `yaml:"travel_api_tvm_id"`
	RequestTimeout        time.Duration `yaml:"request_timeout"`
	ElapsedRequestTimeout time.Duration `yaml:"elapsed_request_timeout"`
}

var DefaultHTTPConfig = Config{
	TravelAPIHost:         "https://api.travel-balancer-test.yandex.net",
	TravelAPITvmID:        2002548,
	RequestTimeout:        1500 * time.Millisecond,
	ElapsedRequestTimeout: 6 * time.Second,
}

type HTTPClient struct {
	backOffPolicyGetter func() backoff.BackOff
	httpClient          *http.Client
	tvmClient           tvm.Client
	config              Config
	metrics             *HTTPMetrics
}

func NewHTTPClient(config Config, httpClient *http.Client, tvmClient tvm.Client, opts ...Option) *HTTPClient {
	c := &HTTPClient{
		config:     config,
		httpClient: httpClient,
		tvmClient:  tvmClient,
		metrics:    NewHTTPMetrics("orders_http_client"),
	}
	defaultOpts := []Option{
		withBackOffPolicy(config.ElapsedRequestTimeout),
	}
	for _, opt := range append(defaultOpts, opts...) {
		opt(c)
	}
	return c
}

func (c *HTTPClient) GetOrdersByIDs(ctx context.Context, orderIDs ...orders.ID) ([]models.OrderItem, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "OrdersClient.GetOrdersByIDs")
	defer tracingSpan.Finish()

	requestBuilder := newOrdersByIDsRequestBuilder(c, orderIDs)
	responseBytes, err := c.getJSON(ctx, requestBuilder)
	if err != nil {
		return nil, err
	}
	result := &selectOrdersRsp{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, err
	}
	return result.Orders, nil
}

func (c *HTTPClient) GetOrderNoAuth(ctx context.Context, orderID orders.ID) (models.OrderItem, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "OrdersClient.GetOrderNoAuth")
	defer tracingSpan.Finish()

	requestBuilder := newGetOrderRequestBuilder(c, orderID)
	responseBytes, err := c.getJSON(ctx, requestBuilder)
	if err != nil {
		return models.OrderItem{}, err
	}
	result := &getOrderRsp{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return models.OrderItem{}, err
	}
	return result.Order, nil
}

func (c *HTTPClient) GetUserOrdersWithoutExcluded(ctx context.Context, _ string, excludedIDs ...orders.ID) ([]models.OrderItem, string, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "OrdersClient.GetUserOrdersWithoutExcluded")
	defer tracingSpan.Finish()

	requestBuilder := newGetOrdersWithoutExcludedRequestBuilder(c, excludedIDs)
	responseBytes, err := c.getJSON(ctx, requestBuilder)
	if err != nil {
		return nil, "", err
	}
	result := &getOrdersWithoutExcludedRsp{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, "", err
	}
	return result.Orders, result.NextPageToken, nil
}

func (c *HTTPClient) GetUserOrdersWithoutExcludedNextPage(
	ctx context.Context,
	nextPageToken string,
) ([]models.OrderItem, string, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "OrdersClient.GetUserOrdersWithoutExcludedNextPage")
	defer tracingSpan.Finish()

	result, err := c.getNextPage(ctx, nextPageToken)
	if err != nil {
		return nil, "", err
	}
	return result.Orders, result.NextPageToken, nil
}

func (c *HTTPClient) getNextPage(ctx context.Context, nextPageToken string) (*getOrdersWithoutExcludedRsp, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "OrdersClient.getNextPage")
	defer tracingSpan.Finish()

	requestBuilder := newGetOrdersWithoutExcludedNextPageRequestBuilder(c, nextPageToken)
	responseBytes, err := c.getJSON(ctx, requestBuilder)
	if err != nil {
		return nil, err
	}
	result := &getOrdersWithoutExcludedRsp{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, err
	}
	return result, nil
}

func (c *HTTPClient) getJSON(ctx context.Context, requestBuilder requestBuilder) (_ []byte, err error) {
	startRequestTime := time.Now()
	defer func() {
		c.metrics.SendRequestMetrics(requestBuilder.GetURLPath(), startRequestTime, err == nil)
	}()

	var byteResponse []byte
	requestFunc := func() error {
		ctx, cancel := context.WithTimeout(ctx, c.config.RequestTimeout)
		defer cancel()

		request, err := requestBuilder.BuildRequest(ctx)
		if err != nil {
			return backoff.Permanent(err)
		}

		response, err := c.httpClient.Do(request)
		if err != nil {
			return RequestTravelAPIErr{err}
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return RequestTravelAPIErr{err}
		}
		if response.StatusCode == http.StatusNotFound {
			return backoff.Permanent(NewTravelAPINotFoundErr(string(byteResponse)))
		}
		if response.StatusCode != http.StatusOK {
			err := fmt.Errorf("travel api answered with code %d: %s", response.StatusCode, string(byteResponse))
			if response.StatusCode == http.StatusBadRequest {
				return backoff.Permanent(err)
			}
			return RequestTravelAPIErr{err}
		}
		return nil
	}
	if err = c.retryRequest(requestFunc); err != nil {
		return nil, RequestTravelAPIErr{err}
	}
	return byteResponse, nil
}

func (c *HTTPClient) setCredentialHeaders(request *http.Request) error {
	userCredentials := usercredentials.FromContext(request.Context())
	yandexUID, ok := userCredentials.YandexUID()
	if !ok {
		return ErrNoUserCredentials
	}
	passportID, ok := userCredentials.PassportID()
	if !ok {
		return ErrNoUserCredentials
	}
	userTicket, ok := userCredentials.UserTicket()
	if !ok {
		return ErrNoUserCredentials
	}
	request.Header.Set(HeaderXYaYandexUID, yandexUID)
	request.Header.Set(HeaderXYaPassportID, passportID)
	request.Header.Set(tvmutil.XYaUserTicket, userTicket)
	err := c.setServiceTicket(request)
	if err != nil {
		return err
	}
	return nil
}

func (c *HTTPClient) setServiceTicket(request *http.Request) error {
	var tvmTicket string
	if c.config.TravelAPITvmID != 0 {
		var err error
		tvmTicket, err = c.tvmClient.GetServiceTicketForID(request.Context(), tvm.ClientID(c.config.TravelAPITvmID))
		if err != nil {
			return err
		}
	}
	if tvmTicket != "" {
		request.Header.Set(tvmutil.XYaServiceTicket, tvmTicket)
	}
	return nil
}

func (c *HTTPClient) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}
