package clients

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/models"
)

type Client interface {
	GetOrdersByIDs(ctx context.Context, orderIDs ...orders.ID) (map[orders.ID]models.OrderItem, error)
	GetUserOrdersWithoutExcluded(ctx context.Context, passportID string, excludedIDs ...orders.ID) (
		map[orders.ID]models.OrderItem,
		error,
	)
}

const clientName = "orders.clients.MappingClient"

type AviaOrderMapper interface {
	Map(item *models.AviaItem) (_ orders.Order, err error)
}

type HotelOrderMapper interface {
	Map(item *models.HotelItem) (_ orders.Order, err error)
}

type TrainOrderMapper interface {
	Map(item *models.TrainItem) (_ orders.Order, err error)
}

type BusOrderMapper interface {
	Map(item *models.BusItem) (_ orders.Order, err error)
}

type MappingClient struct {
	logger           log.Logger
	httpClient       *HTTPClient
	aviaOrderMapper  AviaOrderMapper
	hotelOrderMapper HotelOrderMapper
	trainOrderMapper TrainOrderMapper
	busOrderMapper   BusOrderMapper
}

func NewMappingClient(
	logger log.Logger,
	httpClient *HTTPClient,
	aviaOrderMapper AviaOrderMapper,
	hotelOrderMapper HotelOrderMapper,
	trainOrderMapper TrainOrderMapper,
	busOrderMapper BusOrderMapper,
) *MappingClient {
	return &MappingClient{
		logger:           logger.WithName(clientName),
		httpClient:       httpClient,
		aviaOrderMapper:  aviaOrderMapper,
		hotelOrderMapper: hotelOrderMapper,
		trainOrderMapper: trainOrderMapper,
		busOrderMapper:   busOrderMapper,
	}
}

func (m *MappingClient) GetOrdersByIDs(ctx context.Context, orderIDs ...orders.ID) ([]orders.Order, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, fmt.Sprintf("%s.GetOrdersByIDs", clientName))
	defer tracingSpan.
		SetTag("ordersCount", len(orderIDs)).
		Finish()

	orderItems, err := m.httpClient.GetOrdersByIDs(ctx, orderIDs...)
	if err != nil {
		return nil, err
	}
	return m.mapOrderItems(orderItems)
}

func (m *MappingClient) GetOrderNoAuth(ctx context.Context, orderID orders.ID) (orders.Order, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, fmt.Sprintf("%s.GetOrderNoAuth", clientName))
	defer tracingSpan.
		SetTag("orderID", string(orderID)).
		Finish()

	order, err := m.httpClient.GetOrderNoAuth(ctx, orderID)
	if err != nil {
		return nil, err
	}
	return m.mapOrderItem(order)
}

func (m *MappingClient) GetUserOrdersWithoutExcluded(
	ctx context.Context,
	passportID string,
	excludedIDs ...orders.ID,
) ([]orders.Order, string, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, fmt.Sprintf("%s.GetUserOrdersWithoutExcluded", clientName))
	defer tracingSpan.
		SetTag("passportID", passportID).
		SetTag("ordersCount", len(excludedIDs)).
		Finish()

	orderItems, nextPageToken, err := m.httpClient.GetUserOrdersWithoutExcluded(ctx, passportID, excludedIDs...)
	if err != nil {
		return nil, "", err
	}
	mappedOrders, err := m.mapOrderItems(orderItems)
	return mappedOrders, nextPageToken, err
}

func (m *MappingClient) GetUserOrdersWithoutExcludedNextPage(
	ctx context.Context,
	nextPageToken string,
) ([]orders.Order, string, error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, fmt.Sprintf("%s.GetUserOrdersWithoutExcludedNextPage", clientName))
	defer tracingSpan.Finish()

	orderItems, nextPageToken, err := m.httpClient.GetUserOrdersWithoutExcludedNextPage(ctx, nextPageToken)
	if err != nil {
		return nil, "", err
	}
	mappedOrders, err := m.mapOrderItems(orderItems)
	return mappedOrders, nextPageToken, err
}

func (m *MappingClient) mapOrderItems(orderItems []models.OrderItem) ([]orders.Order, error) {
	result := make([]orders.Order, 0, len(orderItems))
	for _, item := range orderItems {
		mappedOrder, err := m.mapOrderItem(item)
		if err != nil {
			return nil, err
		}
		if mappedOrder != nil {
			result = append(result, mappedOrder)
		}
	}
	return result, nil
}

func (m *MappingClient) mapOrderItem(order models.OrderItem) (orders.Order, error) {
	if order.HasAvia() {
		return m.aviaOrderMapper.Map(order.Avia)
	} else if order.HasHotel() {
		return m.hotelOrderMapper.Map(order.Hotel)
	} else if order.HasTrain() {
		return m.trainOrderMapper.Map(order.Train)
	} else if order.HasBus() {
		return m.busOrderMapper.Map(order.Bus)
	} else {
		m.logger.Warn("got order unsupported order type", log.String("orderID", order.ID()))
		return nil, nil
	}
}
