package clients

import (
	"bytes"
	"context"
	"encoding/json"
	"net/http"
	"net/url"

	"a.yandex-team.ru/library/go/httputil/headers"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
)

type requestBuilder interface {
	GetURLPath() string
	BuildRequest(ctx context.Context) (*http.Request, error)
}

type ordersByIDsRequestBuilder struct {
	client   *HTTPClient
	orderIDs []orders.ID
}

func newOrdersByIDsRequestBuilder(client *HTTPClient, orderIDs []orders.ID) *ordersByIDsRequestBuilder {
	return &ordersByIDsRequestBuilder{client: client, orderIDs: orderIDs}
}

func (b *ordersByIDsRequestBuilder) GetURLPath() string {
	return "/api/trips/orders/v1/select_orders"
}

func (b *ordersByIDsRequestBuilder) BuildRequest(ctx context.Context) (*http.Request, error) {
	requestURL := b.buildURL()
	requestBytes, _ := json.Marshal(selectOrdersReq{emptyOrdersInsteadOfNil(b.orderIDs)})
	request, err := http.NewRequestWithContext(ctx, http.MethodPost, requestURL, bytes.NewReader(requestBytes))
	if err != nil {
		return nil, err
	}
	if err = b.client.setCredentialHeaders(request); err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	return request, nil
}

func (b *ordersByIDsRequestBuilder) buildURL() string {
	return b.client.config.TravelAPIHost + b.GetURLPath()
}

type getOrdersWithoutExcludedRequestBuilder struct {
	client   *HTTPClient
	orderIDs []orders.ID
}

func newGetOrdersWithoutExcludedRequestBuilder(client *HTTPClient, orderIDs []orders.ID) *getOrdersWithoutExcludedRequestBuilder {
	return &getOrdersWithoutExcludedRequestBuilder{client: client, orderIDs: orderIDs}
}

func (b *getOrdersWithoutExcludedRequestBuilder) GetURLPath() string {
	return "/api/trips/orders/v1/get_orders_without_excluded"
}

func (b *getOrdersWithoutExcludedRequestBuilder) BuildRequest(ctx context.Context) (*http.Request, error) {
	requestURL := b.buildURL()
	const pageSize = 150
	requestBytes, _ := json.Marshal(newGetOrdersWithoutExcludedReq(emptyOrdersInsteadOfNil(b.orderIDs), pageSize))
	request, err := http.NewRequestWithContext(ctx, http.MethodPost, requestURL, bytes.NewReader(requestBytes))
	if err != nil {
		return nil, err
	}
	if err = b.client.setCredentialHeaders(request); err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	return request, nil
}

func (b *getOrdersWithoutExcludedRequestBuilder) buildURL() string {
	return b.client.config.TravelAPIHost + b.GetURLPath()
}

type getOrdersWithoutExcludedNextPageRequestBuilder struct {
	client        *HTTPClient
	nextPageToken string
}

func newGetOrdersWithoutExcludedNextPageRequestBuilder(client *HTTPClient, nextPageToken string) *getOrdersWithoutExcludedNextPageRequestBuilder {
	return &getOrdersWithoutExcludedNextPageRequestBuilder{client: client, nextPageToken: nextPageToken}
}

func (b *getOrdersWithoutExcludedNextPageRequestBuilder) GetURLPath() string {
	return "/api/trips/orders/v1/get_orders_without_excluded_next_page"
}

func (b *getOrdersWithoutExcludedNextPageRequestBuilder) BuildRequest(ctx context.Context) (*http.Request, error) {
	requestURL := b.buildURL()
	requestBytes, _ := json.Marshal(getOrdersWithoutExcludedNextPageReq{NextPageToken: b.nextPageToken})
	request, err := http.NewRequestWithContext(
		ctx,
		http.MethodPost,
		requestURL,
		bytes.NewReader(requestBytes),
	)
	if err != nil {
		return nil, err
	}
	if err = b.client.setCredentialHeaders(request); err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	return request, nil
}

func (b *getOrdersWithoutExcludedNextPageRequestBuilder) buildURL() string {
	return b.client.config.TravelAPIHost + b.GetURLPath()
}

type getOrderRequestBuilder struct {
	client  *HTTPClient
	orderID orders.ID
}

func newGetOrderRequestBuilder(client *HTTPClient, orderID orders.ID) *getOrderRequestBuilder {
	return &getOrderRequestBuilder{client: client, orderID: orderID}
}

func (b *getOrderRequestBuilder) GetURLPath() string {
	return "/api/trips/orders/v1/get_order"
}

func (b *getOrderRequestBuilder) BuildRequest(ctx context.Context) (*http.Request, error) {
	requestURL := b.buildURL(b.orderID)
	request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL, nil)
	if err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := b.client.setServiceTicket(request); err != nil {
		return nil, err
	}
	return request, nil
}

func (b *getOrderRequestBuilder) buildURL(id orders.ID) string {
	requestURL, _ := url.Parse(b.client.config.TravelAPIHost + b.GetURLPath())
	query := url.Values{"orderId": []string{id.String()}}
	requestURL.RawQuery = query.Encode()
	return requestURL.String()
}

func emptyOrdersInsteadOfNil(items []orders.ID) []orders.ID {
	if len(items) == 0 {
		return make([]orders.ID, 0)
	}
	return items
}
