package mappers

import (
	"time"

	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	ordermodels "a.yandex-team.ru/travel/komod/trips/internal/orders/models"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

const aviaFlightTimesFormat = "2006-01-02T15:04:05"

type LocationRepository interface {
	LoadLocation(location string) (*time.Location, error)
}

type AviaOrderMapper struct {
	registry                     references.References
	locationRepository           LocationRepository
	settlementByStationExtractor *extractors.SettlementByStationExtractor
}

func NewAviaOrderMapper(
	registry references.References,
	locationRepository LocationRepository,
	settlementByStationExtractor *extractors.SettlementByStationExtractor,
) *AviaOrderMapper {
	return &AviaOrderMapper{
		registry:                     registry,
		locationRepository:           locationRepository,
		settlementByStationExtractor: settlementByStationExtractor,
	}
}

type routePart struct {
	departureStation    *rasp.TStation
	arrivalStation      *rasp.TStation
	departureSettlement *rasp.TSettlement
	arrivalSettlement   *rasp.TSettlement
	departureLocation   *time.Location
	arrivalLocation     *time.Location
	departureTime       time.Time
	arrivalTime         time.Time
}

func (m *AviaOrderMapper) Map(item *ordermodels.AviaItem) (_ orders.Order, err error) {
	defer errutil.Wrap(&err, "orders.mappers.AviaOrderMapper.Map")

	if len(item.AirReservation.Segments) == 0 {
		return nil, ErrUnexpectedSegmentsNumber
	}

	forwardFlights := item.AirReservation.Segments[0].Flights
	forwardRoutePart, err := m.extractRoutePart(forwardFlights)
	if err != nil {
		return nil, err
	}

	var backwardDepartureTime *time.Time
	var backwardArrivalTime *time.Time
	if len(item.AirReservation.Segments) > 1 {
		backwardFlights := item.AirReservation.Segments[len(item.AirReservation.Segments)-1].Flights
		backwardRoutePart, err := m.extractRoutePart(backwardFlights)
		if err != nil {
			return nil, err
		}
		backwardDepartureTime = &backwardRoutePart.departureTime
		backwardArrivalTime = &backwardRoutePart.arrivalTime
	}

	carriers, err := m.extractCarriers(item.AirReservation.Segments)
	if err != nil {
		return nil, err
	}

	return &orders.AviaOrder{
		BaseOrder: orders.NewBaseOrder(
			orders.ID(item.ID),
			item.PassportID,
			item.State.ToProto(),
		),
		Route: orders.Route{
			FromSettlement:    forwardRoutePart.departureSettlement,
			ToSettlement:      forwardRoutePart.arrivalSettlement,
			FromStation:       forwardRoutePart.departureStation,
			ToStation:         forwardRoutePart.arrivalStation,
			ForwardDeparture:  forwardRoutePart.departureTime,
			ForwardArrival:    forwardRoutePart.arrivalTime,
			BackwardDeparture: backwardDepartureTime,
			BackwardArrival:   backwardArrivalTime,
		},
		PNR:      item.AirReservation.Pnr,
		Carriers: carriers,
	}, nil
}

func (m *AviaOrderMapper) extractRoutePart(flights []ordermodels.Flight) (*routePart, error) {
	if len(flights) == 0 {
		return nil, ErrUnexpectedFlightsNumber
	}

	departureStationID := m.extractDepartureStationID(flights)
	departureStation, found := m.registry.Stations().Get(departureStationID)
	if !found {
		return nil, NewErrStationNotFound(departureStationID)
	}
	departureSettlement, _ := m.settlementByStationExtractor.Get(departureStation.Id)

	arrivalStationID := m.extractArrivalStationID(flights)
	arrivalStation, found := m.registry.Stations().Get(arrivalStationID)
	if !found {
		return nil, NewErrStationNotFound(arrivalStationID)
	}
	arrivalSettlement, _ := m.settlementByStationExtractor.Get(arrivalStation.Id)

	departureLocation, err := m.locationRepository.LoadLocation(departureStation.TimeZoneCode)
	if err != nil {
		return nil, err
	}
	arrivalLocation, err := m.locationRepository.LoadLocation(arrivalStation.TimeZoneCode)
	if err != nil {
		return nil, err
	}

	departureTime, err := m.extractDepartureTime(flights, departureLocation)
	if err != nil {
		return nil, err
	}
	arrivalTime, err := m.extractArrivalTime(flights, arrivalLocation)
	if err != nil {
		return nil, err
	}

	return &routePart{
		departureStation:    departureStation,
		arrivalStation:      arrivalStation,
		departureSettlement: departureSettlement,
		arrivalSettlement:   arrivalSettlement,
		departureLocation:   departureLocation,
		arrivalLocation:     arrivalLocation,
		departureTime:       departureTime,
		arrivalTime:         arrivalTime,
	}, nil
}

func (m *AviaOrderMapper) extractDepartureTime(flights []ordermodels.Flight, location *time.Location) (time.Time, error) {
	return time.ParseInLocation(aviaFlightTimesFormat, flights[0].Departure, location)

}

func (m *AviaOrderMapper) extractArrivalTime(flights []ordermodels.Flight, location *time.Location) (time.Time, error) {
	return time.ParseInLocation(aviaFlightTimesFormat, flights[len(flights)-1].Arrival, location)
}

func (m *AviaOrderMapper) extractDepartureStationID(flights []ordermodels.Flight) int {
	return int(flights[0].From)
}

func (m *AviaOrderMapper) extractArrivalStationID(flights []ordermodels.Flight) int {
	return int(flights[len(flights)-1].To)
}

func (m *AviaOrderMapper) extractCarriers(segments []ordermodels.AviaSegment) ([]*rasp.TCarrier, error) {
	knownIDs := make(map[int]bool)
	carriers := make([]*rasp.TCarrier, 0)
	for _, segment := range segments {
		for _, flight := range segment.Flights {
			carrierID := int(flight.OperatingAviaCompany)
			if !knownIDs[carrierID] {
				knownIDs[carrierID] = true
				carrier, found := m.registry.Carriers().Get(carrierID)
				if !found {
					return nil, NewErrCarrierNotFound(carrierID)
				}
				carriers = append(carriers, carrier)
			}
		}
	}
	return carriers, nil
}
