package mappers

import (
	"fmt"
	"time"

	"github.com/go-openapi/strfmt"

	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	ordermodels "a.yandex-team.ru/travel/komod/trips/internal/orders/models"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/models/buses"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type BusOrderMapper struct {
	registry                     references.References
	locationRepository           LocationRepository
	settlementByStationExtractor *extractors.SettlementByStationExtractor
}

func NewBusOrderMapper(
	registry references.References,
	locationRepository LocationRepository,
	settlementByStationExtractor *extractors.SettlementByStationExtractor,
) *BusOrderMapper {
	return &BusOrderMapper{
		registry:                     registry,
		locationRepository:           locationRepository,
		settlementByStationExtractor: settlementByStationExtractor,
	}
}

func (m *BusOrderMapper) Map(item *ordermodels.BusItem) (_ orders.Order, err error) {
	defer errutil.Wrap(&err, "orders.mappers.BusOrderMapper.Map")
	if len(item.Rides) == 0 {
		return nil, ErrNoBusRides{item.ID}
	}
	ride := item.Rides[0]

	departureTime, err := m.parseTime(ride.LocalDepartureTime, ride.TitlePointFrom)
	if err != nil {
		return nil, err
	}
	arrivalTime, err := m.parseTime(ride.LocalArrivalTime, ride.TitlePointTo)
	if err != nil {
		return nil, err
	}
	fromSettlement := m.parseSettlement(ride.TitlePointFrom)
	toSettlement := m.parseSettlement(ride.TitlePointTo)
	return &orders.BusOrder{
		BaseOrder: orders.NewBaseOrder(
			orders.ID(item.ID),
			item.PassportID,
			item.State.ToProto(),
		),
		DownloadBlankToken: ride.DownloadBlankToken,
		Route: orders.Route{
			FromSettlement:    fromSettlement,
			ToSettlement:      toSettlement,
			FromStation:       m.parseStation(ride.TitlePointFrom),
			ToStation:         m.parseStation(ride.TitlePointTo),
			ForwardDeparture:  departureTime,
			ForwardArrival:    arrivalTime,
			BackwardDeparture: nil,
			BackwardArrival:   nil,
		},
		Title:                m.buildTitle(ride.TitlePointFrom, ride.TitlePointTo),
		Description:          m.buildRouteDescription(ride.TitlePointFrom, ride.TitlePointTo),
		RefundedTicketsCount: ride.RefundedTicketsCount,
		CarrierName:          ride.Carrier.Name,
	}, nil
}

func (m *BusOrderMapper) parseSettlement(point buses.Point) *rasp.TSettlement {
	switch point.Type {
	case buses.PointTypeStation:
		if settlement, ok := m.settlementByStationExtractor.Get(int32(point.ID)); ok {
			return settlement
		}
	case buses.PointTypeSettlement:
		if settlement, ok := m.registry.Settlements().Get(point.ID); ok {
			return settlement
		}
	}
	return nil
}

func (m *BusOrderMapper) parseTime(rawTime string, fromPoint buses.Point) (time.Time, error) {
	location, err := m.locationRepository.LoadLocation(fromPoint.Timezone)
	if err != nil {
		return time.Time{}, err
	}
	return time.ParseInLocation(strfmt.ISO8601LocalTime, rawTime, location)
}

func (m *BusOrderMapper) buildRouteDescription(pointFrom buses.Point, pointTo buses.Point) string {
	return fmt.Sprintf(
		"%s — %s",
		strutil.Coalesce(pointFrom.SupplierDescription, pointFrom.Title),
		strutil.Coalesce(pointTo.SupplierDescription, pointTo.Title),
	)
}

func (m *BusOrderMapper) parseStation(point buses.Point) *rasp.TStation {
	switch point.Type {
	case buses.PointTypeStation:
		if station, ok := m.registry.Stations().Get(point.ID); ok {
			return station
		}
		return nil
	default:
		return nil
	}
}

func (m *BusOrderMapper) buildTitle(pointFrom, pointTo buses.Point) string {
	return fmt.Sprintf("%s — %s", pointFrom.Title, pointTo.Title)
}
