package mappers

import (
	"time"

	timeformats "cuelang.org/go/pkg/time"
	"github.com/shopspring/decimal"

	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	ordermodels "a.yandex-team.ru/travel/komod/trips/internal/orders/models"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/library/go/geobase"
	travelproto "a.yandex-team.ru/travel/proto"
)

type HotelOrderMapper struct {
	geoBase          geobase.Geobase
	registry         references.References
	spanPointFactory *point.Factory
}

func NewHotelOrderMapper(
	geoBase geobase.Geobase,
	registry references.References,
	spanPointFactory *point.Factory,
) *HotelOrderMapper {
	return &HotelOrderMapper{
		geoBase:          geoBase,
		registry:         registry,
		spanPointFactory: spanPointFactory,
	}
}

func (m *HotelOrderMapper) Map(item *ordermodels.HotelItem) (orders.Order, error) {
	point, err := m.spanPointFactory.MakeByGeoID(int(item.CityGeoID))
	if err != nil {
		point, err = m.spanPointFactory.MakeByGeoID(int(item.GeoID))
		if err != nil {
			return nil, err
		}
	}

	return &orders.HotelOrder{
		BaseOrder: orders.NewBaseOrder(
			orders.ID(item.ID),
			item.PassportID,
			item.State.ToProto(),
		),
		Point:        point,
		CityGeoID:    int(item.CityGeoID),
		Title:        item.HotelName,
		Stars:        uint32(item.Stars),
		CheckinDate:  parseHotelDate(item.CheckinDate),
		CheckoutDate: parseHotelDate(item.CheckoutDate),
		Address:      item.Address,
		Coordinates: &travelproto.TCoordinates{
			Latitude:  item.Coordinates.Latitude,
			Longitude: item.Coordinates.Longitude,
		},
		ImageURLTemplate:   item.ImageURLTemplate,
		DocumentURL:        item.DocumentURL,
		HasDeferredPayment: item.Payment != nil && item.Payment.Next.Amount.Value.GreaterThan(decimal.Zero),
	}, nil
}

func parseHotelDate(hotelDate string) time.Time {
	date, _ := time.Parse(timeformats.RFC3339Date, hotelDate)
	return date
}
