package mappers

import (
	"fmt"
	"sort"
	"time"

	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	ordermodels "a.yandex-team.ru/travel/komod/trips/internal/orders/models"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/library/go/errutil"
)

const trainTimesFormat = time.RFC3339

type TrainOrderMapper struct {
	registry                     references.References
	locationRepository           LocationRepository
	settlementByStationExtractor *extractors.SettlementByStationExtractor
}

func NewTrainOrderMapper(
	registry references.References,
	locationRepository LocationRepository,
	settlementByStationExtractor *extractors.SettlementByStationExtractor,
) *TrainOrderMapper {
	return &TrainOrderMapper{
		registry:                     registry,
		locationRepository:           locationRepository,
		settlementByStationExtractor: settlementByStationExtractor,
	}
}

func (m *TrainOrderMapper) Map(item *ordermodels.TrainItem) (_ orders.Order, err error) {
	defer errutil.Wrap(&err, "orders.mappers.TrainOrderMapper.Map")

	forwardSegments := filterTrainSegmentsByDirection(ordermodels.TrainDirectionForward, item.OrderInfo.Segments)
	backwardSegments := filterTrainSegmentsByDirection(ordermodels.TrainDirectionBackward, item.OrderInfo.Segments)
	sort.Sort(trainSegmentsByDeparture(forwardSegments))
	sort.Sort(trainSegmentsByDeparture(backwardSegments))

	if err := validateSegments(item.ID, item.OrderInfo.Segments...); err != nil {
		return nil, err
	}

	forwardRoutePart, err := m.extractRoutePart(forwardSegments)
	if err != nil {
		return nil, err
	}

	var backwardDepartureTime *time.Time
	var backwardArrivalTime *time.Time
	if len(backwardSegments) != 0 {
		backwardRoutePart, err := m.extractRoutePart(backwardSegments)
		if err != nil {
			return nil, err
		}
		backwardDepartureTime = &backwardRoutePart.departureTime
		backwardArrivalTime = &backwardRoutePart.arrivalTime
	}

	trains, err := m.extractTrains(item.OrderInfo.Segments)
	if err != nil {
		return nil, err
	}

	return &orders.TrainOrder{
		BaseOrder: orders.NewBaseOrder(
			orders.ID(item.ID),
			item.PassportID,
			item.State.ToProto(),
		),
		Route: orders.Route{
			FromSettlement:    forwardRoutePart.departureSettlement,
			ToSettlement:      forwardRoutePart.arrivalSettlement,
			FromStation:       forwardRoutePart.departureStation,
			ToStation:         forwardRoutePart.arrivalStation,
			ForwardDeparture:  forwardRoutePart.departureTime,
			ForwardArrival:    forwardRoutePart.arrivalTime,
			BackwardDeparture: backwardDepartureTime,
			BackwardArrival:   backwardArrivalTime,
		},
		RefundedTicketsCount: m.countRefundedTickets(item.OrderInfo.Segments),
		Trains:               trains,
	}, nil
}

func (m *TrainOrderMapper) extractRoutePart(segments []ordermodels.TrainSegmentInfo) (*routePart, error) {
	if len(segments) == 0 {
		return nil, ErrUnexpectedSegmentsNumber
	}

	firstSegment := segments[0]
	lastSegment := segments[len(segments)-1]

	departureStation, found := m.registry.Stations().Get(firstSegment.StationFrom.ID)
	if !found {
		return nil, NewErrStationNotFound(firstSegment.StationFrom.ID)
	}

	arrivalStation, found := m.registry.Stations().Get(lastSegment.StationTo.ID)
	if !found {
		return nil, NewErrStationNotFound(lastSegment.StationTo.ID)
	}

	departureSettlement, _ := m.settlementByStationExtractor.Get(departureStation.Id)
	arrivalSettlement, _ := m.settlementByStationExtractor.Get(arrivalStation.Id)

	departureLocation, err := m.locationRepository.LoadLocation(departureStation.TimeZoneCode)
	if err != nil {
		return nil, err
	}
	departureTime, err := time.Parse(trainTimesFormat, firstSegment.Departure)
	if err != nil {
		return nil, err
	}
	departureTime = departureTime.In(departureLocation)

	arrivalLocation, err := m.locationRepository.LoadLocation(arrivalStation.TimeZoneCode)
	if err != nil {
		return nil, err
	}
	arrivalTime, err := time.Parse(trainTimesFormat, lastSegment.Arrival)
	if err != nil {
		return nil, err
	}
	arrivalTime = arrivalTime.In(arrivalLocation)
	return &routePart{
		departureStation:    departureStation,
		arrivalStation:      arrivalStation,
		departureSettlement: departureSettlement,
		arrivalSettlement:   arrivalSettlement,
		departureLocation:   departureLocation,
		arrivalLocation:     arrivalLocation,
		departureTime:       departureTime,
		arrivalTime:         arrivalTime,
	}, nil
}

func validateSegments(orderID string, segments ...ordermodels.TrainSegmentInfo) error {
	for _, s := range segments {
		if s.Departure == "" {
			return orders.NewErrOrderValidation(fmt.Sprintf("train order %s with empty departure time", orderID))
		}
		if s.Arrival == "" {
			return orders.NewErrOrderValidation(fmt.Sprintf("train order %s with empty arrival time", orderID))
		}
	}
	return nil
}

func (m *TrainOrderMapper) extractTrains(segments []ordermodels.TrainSegmentInfo) (result []*orders.Train, err error) {
	stationReference := m.registry.Stations()
	for _, segment := range segments {
		fromStationID := segment.StationFrom.ID
		fromStation, found := stationReference.Get(fromStationID)
		if !found {
			return nil, NewErrStationNotFound(fromStationID)
		}
		fromSettlement, _ := m.settlementByStationExtractor.Get(fromStation.Id)

		toStationID := segment.StationTo.ID
		toStation, found := stationReference.Get(toStationID)
		if !found {
			return nil, NewErrStationNotFound(toStationID)
		}
		toSettlement, _ := m.settlementByStationExtractor.Get(toStation.Id)

		result = append(
			result,
			&orders.Train{
				TrainInfo: orders.TrainInfo{
					Direction:            orders.TrainDirection(segment.Direction),
					Number:               segment.TrainInfo.TrainNumber,
					BrandTitle:           segment.TrainInfo.BrandTitle,
					StartSettlementTitle: segment.TrainInfo.StartSettlementTitle,
					EndSettlementTitle:   segment.TrainInfo.EndSettlementTitle,
				},
				FromSettlement: fromSettlement,
				ToSettlement:   toSettlement,
				FromStation:    fromStation,
				ToStation:      toStation,
			},
		)
	}
	return result, nil
}

func (m *TrainOrderMapper) countRefundedTickets(segments []ordermodels.TrainSegmentInfo) uint16 {
	result := uint16(0)
	for _, segment := range segments {
		for _, passenger := range segment.Passengers {
			if passenger.TicketRzhdStatus == ordermodels.TicketRzhdStatusRefunded &&
				passenger.Category != ordermodels.TrainPassengerCategoryBabyWithoutPlace {
				result++
			}
		}
	}
	return result
}

func filterTrainSegmentsByDirection(
	direction ordermodels.TrainDirection,
	segments []ordermodels.TrainSegmentInfo,
) []ordermodels.TrainSegmentInfo {
	filtered := make([]ordermodels.TrainSegmentInfo, 0)
	for _, segment := range segments {
		if segment.Direction == direction {
			filtered = append(filtered, segment)
		}
	}
	return filtered
}

type trainSegmentsByDeparture []ordermodels.TrainSegmentInfo

func (s trainSegmentsByDeparture) Len() int {
	return len(s)
}

func (s trainSegmentsByDeparture) Less(i, j int) bool {
	return s[i].Departure < s[j].Departure
}

func (s trainSegmentsByDeparture) Swap(i, j int) {
	s[i], s[j] = s[j], s[i]
}
