package orders

import (
	"time"

	tripsapi "a.yandex-team.ru/travel/komod/trips/api/trips/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	ordersproto "a.yandex-team.ru/travel/orders/proto"
	travelproto "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type Order interface {
	isOrder()
	ID() ID
	PassportID() string
	State() ordersproto.EDisplayOrderState
	TripsOrderState() tripsapi.TripOrderState
	Cancelled() bool
	Refunded() bool
	InProgress() bool
}

type BaseOrder struct {
	id         ID
	passportID string
	state      ordersproto.EDisplayOrderState
}

func NewBaseOrder(id ID, passportID string, state ordersproto.EDisplayOrderState) BaseOrder {
	return BaseOrder{id: id, passportID: passportID, state: state}
}

func (o *BaseOrder) isOrder() {}

func (o *BaseOrder) ID() ID {
	return o.id
}

func (o *BaseOrder) PassportID() string {
	return o.passportID
}

func (o *BaseOrder) Refunded() bool {
	return o.state == ordersproto.EDisplayOrderState_OS_REFUNDED
}

func (o *BaseOrder) Cancelled() bool {
	return o.state == ordersproto.EDisplayOrderState_OS_CANCELLED
}

func (o *BaseOrder) State() ordersproto.EDisplayOrderState {
	return o.state
}

func (o *BaseOrder) TripsOrderState() tripsapi.TripOrderState {
	if o.state == ordersproto.EDisplayOrderState_OS_REFUNDED || o.state == ordersproto.EDisplayOrderState_OS_CANCELLED {
		return tripsapi.TripOrderState_TRIP_ORDER_STATE_CANCELLED
	}
	return tripsapi.TripOrderState_TRIP_ORDER_STATE_CONFIRMED
}

func (o *BaseOrder) InProgress() bool {
	return o.state == ordersproto.EDisplayOrderState_OS_IN_PROGRESS ||
		o.state == ordersproto.EDisplayOrderState_OS_AWAITS_PAYMENT
}

type Route struct {
	FromSettlement    *rasp.TSettlement
	ToSettlement      *rasp.TSettlement
	ForwardDeparture  time.Time
	ForwardArrival    time.Time
	BackwardDeparture *time.Time
	BackwardArrival   *time.Time
	FromStation       *rasp.TStation
	ToStation         *rasp.TStation
}

type AviaOrder struct {
	BaseOrder
	Route
	PNR      string
	Carriers []*rasp.TCarrier
}

type TrainOrder struct {
	BaseOrder
	Route
	Trains               []*Train
	PrintURL             string
	RefundedTicketsCount uint16
}

type BusOrder struct {
	BaseOrder
	Route
	DownloadBlankToken   string
	CarrierName          string
	Title                string
	Description          string
	RefundedTicketsCount uint16
}

type Train struct {
	TrainInfo TrainInfo

	FromSettlement *rasp.TSettlement
	ToSettlement   *rasp.TSettlement

	FromStation *rasp.TStation
	ToStation   *rasp.TStation
}

type TrainInfo struct {
	Direction            TrainDirection
	Number               string
	BrandTitle           string
	StartSettlementTitle string
	EndSettlementTitle   string
}

type TrainDirection string

const (
	TrainDirectionForward  TrainDirection = "FORWARD"
	TrainDirectionBackward TrainDirection = "BACKWARD"
)

type HotelOrder struct {
	BaseOrder
	Point              models.Point
	CityGeoID          int
	Title              string
	Stars              uint32
	CheckinDate        time.Time
	CheckoutDate       time.Time
	Address            string
	Coordinates        *travelproto.TCoordinates
	ImageURLTemplate   string
	DocumentURL        string
	HasDeferredPayment bool
}
