package pgclient

import (
	"context"
	"fmt"
	"time"

	"github.com/jackc/pgx/v4"
	"github.com/jackc/pgx/v4/stdlib"
	"golang.yandex/hasql"

	"a.yandex-team.ru/travel/library/go/errutil"
)

type ClientBuilder struct {
	hosts                  []string
	port                   int
	dbName, user, password string
	initTimeout            time.Duration
	clusterOptions         []hasql.ClusterOption
	poolOptions            []PoolOption
	clientOptions          []ClientOption
}

func NewClientBuilder(
	hosts []string,
	port int,
	dbName string,
	user string,
	password string,
) ClientBuilder {
	return ClientBuilder{
		hosts:          hosts,
		port:           port,
		dbName:         dbName,
		user:           user,
		password:       password,
		initTimeout:    DefaultInitTimeout,
		poolOptions:    defaultPoolOptions,
		clusterOptions: defaultClusterOptions,
		clientOptions:  []ClientOption{WithOnCheckedNode(OnCheckedNode)},
	}
}

func (b ClientBuilder) WithInitTimeout(t time.Duration) ClientBuilder {
	b.initTimeout = t
	return b
}

func (b ClientBuilder) WithClusterOptions(opts ...hasql.ClusterOption) ClientBuilder {
	b.clusterOptions = append(b.clusterOptions, opts...)
	return b
}

func (b ClientBuilder) WithPoolOptions(opts ...PoolOption) ClientBuilder {
	b.poolOptions = append(b.poolOptions, opts...)
	return b
}

func (b ClientBuilder) Build() (client *Client, err error) {
	defer errutil.Wrap(&err, "pgclient.ClientBuilder.Build")

	var nodes []hasql.Node
	for _, host := range b.hosts {
		node, err := b.buildNode(host)
		if err != nil {
			return nil, err
		}
		nodes = append(nodes, node)
	}

	client = &Client{}
	cluster, err := hasql.NewCluster(nodes, client.nodeChecker, b.clusterOptions...)
	if err != nil {
		return nil, err
	}
	client.cluster = cluster

	ctx, cancel := context.WithTimeout(context.Background(), b.initTimeout)
	defer cancel()
	if _, err := cluster.WaitForPrimary(ctx); err != nil {
		return nil, err
	}

	for _, opt := range b.clientOptions {
		opt(client)
	}
	return client, nil
}

func (b ClientBuilder) buildNode(host string) (hasql.Node, error) {
	connString := fmt.Sprintf(
		"host=%s port=%d user=%s password=%s dbname=%s",
		host,
		b.port,
		b.user,
		b.password,
		b.dbName,
	)
	connConfig, err := pgx.ParseConfig(connString)
	if err != nil {
		return nil, err
	}

	// workaround for https://github.com/jackc/pgx/issues/602
	connConfig.BuildStatementCache = nil
	connConfig.PreferSimpleProtocol = true

	db := stdlib.OpenDB(*connConfig)
	for _, opt := range b.poolOptions {
		opt(db)
	}
	return hasql.NewNode(host, db), nil
}
