package pgclient

import (
	"database/sql"
	"errors"
	"time"

	"golang.yandex/hasql"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
)

var (
	ErrPrimaryNodeIsUnavailable = errors.New("postgres: primary node is unavailable")
	ErrNodeIsUnavailable        = errors.New("postgres: node is unavailable")
)

type Client struct {
	cluster       *hasql.Cluster
	onCheckedNode func(*sql.DB, time.Duration)
	logLevel      logger.LogLevel
}

func (c *Client) GetDB(nodeState hasql.NodeStateCriteria) (*gorm.DB, error) {
	if node := c.cluster.Node(nodeState); node != nil {
		return gormWrapper(node)
	}
	switch nodeState {
	case hasql.Primary:
		return nil, ErrPrimaryNodeIsUnavailable
	default:
		return nil, ErrNodeIsUnavailable
	}
}

func (c *Client) ExecuteInTransaction(
	nodeState hasql.NodeStateCriteria,
	processFunc func(*gorm.DB) error,
) error {
	db, err := c.GetDB(nodeState)
	if err != nil {
		return err
	}
	return db.Transaction(processFunc)
}

func (c *Client) Ping() error {
	gormDB, err := c.GetDB(hasql.Primary)
	if err != nil {
		return err
	}
	sqlDB, err := gormDB.DB()
	if err != nil {
		return err
	}
	return sqlDB.Ping()
}
