package pgclient

import (
	"database/sql"
	"time"

	"golang.yandex/hasql"

	"a.yandex-team.ru/library/go/core/log"
	metricscore "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"
)

var (
	latencyBuckets = metricscore.MakeExponentialDurationBuckets(time.Millisecond, 1.3, 35)
)

const (
	databaseMetricsPrefix        = "database"
	databaseStatsMetricsPrefix   = "database.stats"
	aliveNodesMetricName         = "alive"
	primaryNodesMetricName       = "primary"
	nodeLatencyMetricName        = "latency"
	maxOpenConnectionsMetricName = "max_open_connections"
	idleConnectionsMetricName    = "idle_connections"
	inUseConnectionsMetricName   = "in_use_connections"
)

func OnDeadNode(logger log.Logger) func(hasql.Node, error) {
	return func(node hasql.Node, err error) {
		logger.Info("node is dead", log.Any("node", node), log.Error(err))
	}
}

func OnUpdatedNodes(nodes hasql.AliveNodes) {
	for _, node := range nodes.Alive {
		metrics.GlobalAppMetrics().GetOrCreateCounter(
			databaseMetricsPrefix,
			getTags(node.String()),
			aliveNodesMetricName,
		).Inc()
	}
	for _, node := range nodes.Primaries {
		metrics.GlobalAppMetrics().GetOrCreateCounter(
			databaseMetricsPrefix,
			getTags(node.String()),
			primaryNodesMetricName,
		).Inc()
	}
}

func OnCheckedNode(node *sql.DB, latency time.Duration) {
	nodeAddr := getNodeAddr(node)
	tags := getTags(nodeAddr)
	metrics.GlobalAppMetrics().GetOrCreateHistogram(
		databaseMetricsPrefix,
		tags,
		nodeLatencyMetricName,
		latencyBuckets,
	).RecordDuration(latency)

	stats := node.Stats()
	metrics.GlobalAppMetrics().GetOrCreateGauge(databaseStatsMetricsPrefix, tags, idleConnectionsMetricName).Set(float64(stats.Idle))
	metrics.GlobalAppMetrics().GetOrCreateGauge(databaseStatsMetricsPrefix, tags, inUseConnectionsMetricName).Set(float64(stats.InUse))
	metrics.GlobalAppMetrics().GetOrCreateGauge(
		databaseStatsMetricsPrefix,
		tags,
		maxOpenConnectionsMetricName,
	).Set(float64(stats.MaxOpenConnections))
}

func getTags(nodeAddr string) map[string]string {
	return map[string]string{"node": nodeAddr}
}
