package point

import (
	"fmt"
	"strconv"

	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type stationIDToSettlementIDMapper interface {
	Map(stationID int32) (int32, bool)
}

const (
	SochiGeoID  = 239
	SiriusGeoID = 218709
)

// Factory обеспечивает приведение к геобазе если это возможно
type Factory struct {
	stationIDToSettlementIDMapper stationIDToSettlementIDMapper
	cachedLocationRepository      *helpers.CachedLocationRepository
	geoBase                       geobase.Geobase
	references                    references.References
}

func NewFactory(
	stationIDToSettlementIDMapper stationIDToSettlementIDMapper,
	cachedLocationRepository *helpers.CachedLocationRepository,
	geoBase geobase.Geobase,
	references references.References,
) *Factory {
	return &Factory{
		stationIDToSettlementIDMapper: stationIDToSettlementIDMapper,
		cachedLocationRepository:      cachedLocationRepository,
		geoBase:                       geoBase,
		references:                    references,
	}
}

func (f Factory) MakeByPointKey(pointKey string) (models.Point, error) {
	if len(pointKey) == 0 {
		return nil, fmt.Errorf("unexpected pointKey=%s", pointKey)
	}

	id, err := strconv.Atoi(pointKey[1:])
	if err != nil {
		return nil, fmt.Errorf("unexpected pointKey=%s", pointKey)
	}

	switch pointKey[0:1] {
	case "c":
		return f.MakeBySettlementID(id)
	case "s":
		return f.MakeByStationID(id)
	case "g":
		return f.MakeByGeoID(id)
	default:
		return nil, fmt.Errorf("unsupported point type %s", pointKey)
	}
}

func (f Factory) MakeByStationID(stationID int) (models.Point, error) {
	settlementID, found := f.stationIDToSettlementIDMapper.Map(int32(stationID))
	if found {
		return f.MakeBySettlementID(int(settlementID))
	}
	station, found := f.references.Stations().Get(stationID)
	if !found {
		return nil, fmt.Errorf("unknown stationID=%d", stationID)
	}
	return f.makeByStationWithoutSettlement(station)
}

func (f Factory) MakeBySettlementID(settlementID int) (models.Point, error) {
	settlement, found := f.references.Settlements().Get(settlementID)
	if !found {
		return nil, fmt.Errorf("settlement not found: %d", settlementID)
	}

	if settlement.GeoId == 0 {
		linguistics := models.Linguistics{
			models.NominativeCase: strutil.Coalesce(
				settlement.GetTitle().GetRu().GetNominative(),
				settlement.GetTitleDefault(),
			),
			models.GenitiveCase:      settlement.GetTitle().GetRu().GetNominative(),
			models.AccusativeCase:    settlement.GetTitle().GetRu().GetAccusative(),
			models.PrepositionalCase: settlement.GetTitle().GetRu().GetPrepositional(),
			models.Preposition:       settlement.GetTitleRuPrepositionalCase(),
		}
		return models.NewSettlementPoint(settlementID, nil, linguistics), nil
	}
	return f.MakeByGeoID(int(settlement.GeoId))
}

func (f Factory) MakeByRegionID(regionID int) (models.Point, error) {
	region, found := f.references.Regions().Get(regionID)
	if !found {
		return nil, fmt.Errorf("region not found: %d", regionID)
	}
	if region.GeoId == 0 {
		linguistics := models.Linguistics{
			models.NominativeCase: strutil.Coalesce(
				region.GetTitleNominative().GetRu(),
				region.GetTitleDefault(),
			),
		}
		return models.NewRegionPoint(regionID, linguistics), nil
	}
	return f.MakeByGeoID(int(region.GeoId))
}

func (f Factory) MakeByCountryID(countryID int) (models.Point, error) {
	country, found := f.references.Countries().Get(countryID)
	if !found {
		return nil, fmt.Errorf("country not found: %d", countryID)
	}
	if country.GeoId == 0 {

		linguistics := models.Linguistics{
			models.NominativeCase: strutil.Coalesce(
				country.GetTitle().GetRu().GetNominative(),
				country.GetTitleDefault(),
			),
			models.GenitiveCase:      country.GetTitle().GetRu().GetGenitive(),
			models.AccusativeCase:    country.GetTitle().GetRu().GetAccusative(),
			models.PrepositionalCase: country.GetTitle().GetRu().GetPrepositional(),
			models.Preposition:       country.GetTitleDefault(),
		}

		return models.NewCountryPoint(countryID, linguistics), nil
	}
	return f.MakeByGeoID(int(country.GeoId))
}

func (f Factory) MakeBySettlement(settlement *rasp.TSettlement) models.Point {
	point, _ := f.MakeByGeoID(int(settlement.GeoId))
	return point
}

func (f Factory) MakeByGeoID(geoID int) (models.Point, error) {
	if geoID == 0 {
		return nil, fmt.Errorf("zero geoid")
	}
	if geoID == SiriusGeoID {
		geoID = SochiGeoID
	}
	geoLinguistics, err := f.geoBase.GetLinguistics(geoID, consts.Ru)
	if err != nil {
		return nil, err
	}
	region, regErr := f.geoBase.GetRegion(geoID)
	if regErr != nil {
		return nil, regErr
	}

	linguistics := models.Linguistics{
		models.NominativeCase:    geoLinguistics.NominativeCase,
		models.GenitiveCase:      geoLinguistics.GenitiveCase,
		models.AccusativeCase:    geoLinguistics.AccusativeCase,
		models.PrepositionalCase: geoLinguistics.PrepositionalCase,
		models.Preposition:       geoLinguistics.Preposition,
	}

	timezone, tzErr := f.cachedLocationRepository.LoadLocation(region.TimezoneName)
	if tzErr != nil {
		return nil, tzErr
	}
	return models.NewGeoRegionPoint(geoID, timezone, linguistics), nil
}

func ExtractPointKey(point models.Point) string {
	return point.GetPointKey()
}

func (f Factory) makeByStationWithoutSettlement(station *rasp.TStation) (models.Point, error) {
	linguistics := models.Linguistics{
		models.NominativeCase: strutil.Coalesce(
			station.GetTitle().GetRu(),
			station.GetTitleRuNominativeCase(),
			station.GetTitleDefault(),
		),
		models.GenitiveCase:      station.GetTitleRuGenitiveCase(),
		models.AccusativeCase:    station.GetTitleRuAccusativeCase(),
		models.PrepositionalCase: station.GetTitleRuPrepositionalCase(),
		models.Preposition:       station.GetTitleRuPreposition(),
	}

	if station.Latitude == 0 || station.Longitude == 0 {
		return models.NewStationPoint(
			int(station.Id),
			0,
			nil,
			linguistics,
		), nil
	}
	geoRegion, err := f.geoBase.GetRegionByLocation(station.Latitude, station.Longitude)
	if err != nil {
		return nil, err
	}

	timezone, tzErr := f.cachedLocationRepository.LoadLocation(geoRegion.TimezoneName)
	if tzErr != nil {
		return nil, tzErr
	}

	return models.NewStationPoint(
		int(station.Id),
		int(geoRegion.ID),
		timezone,
		linguistics,
	), nil
}
