package references

import (
	"fmt"
	"strings"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type CarrierRepository struct {
	baseRepository *repository.CarrierRepository
	byIata         map[string]*rasp.TCarrier
	bySirena       map[string]*rasp.TCarrier
}

func NewCarrierRepository() *CarrierRepository {
	return &CarrierRepository{
		baseRepository: repository.NewCarrierRepository(),
		byIata:         make(map[string]*rasp.TCarrier),
		bySirena:       make(map[string]*rasp.TCarrier),
	}
}

func (cr *CarrierRepository) Write(b []byte) (int, error) {
	carrier := &rasp.TCarrier{}
	if err := proto.Unmarshal(b, carrier); err != nil {
		return 0, fmt.Errorf("CarrierRepository:Write: %w", err)
	}
	cr.baseRepository.Add(carrier)
	if carrier.Iata != "" {
		cr.byIata[strings.ToUpper(carrier.Iata)] = carrier
	}
	if carrier.SirenaId != "" {
		cr.bySirena[strings.ToUpper(carrier.SirenaId)] = carrier
	}
	return len(b), nil
}

func (cr *CarrierRepository) Get(id int) (*rasp.TCarrier, bool) {
	carrier, ok := cr.baseRepository.Get(id)
	return carrier, ok
}

func (cr *CarrierRepository) GetByCode(code string) (*rasp.TCarrier, bool) {
	upperCode := strings.ToUpper(code)
	if carrier, ok := cr.byIata[upperCode]; ok {
		return carrier, ok
	}
	if carrier, ok := cr.bySirena[upperCode]; ok {
		return carrier, ok
	}
	return nil, false
}

func (cr *CarrierRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewCarrierRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
