package references

import (
	"fmt"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type CountryRepository struct {
	baseRepository *repository.CountryRepository
	byGeoID        map[int]*rasp.TCountry
}

func NewCountryRepository() *CountryRepository {
	return &CountryRepository{
		baseRepository: repository.NewCountryRepository(),
		byGeoID:        make(map[int]*rasp.TCountry),
	}
}

func (cr *CountryRepository) Write(b []byte) (int, error) {
	country := &rasp.TCountry{}
	if err := proto.Unmarshal(b, country); err != nil {
		return 0, fmt.Errorf("CountryRepository:Write: %w", err)
	}
	cr.baseRepository.Add(country)
	if country.GeoId != 0 {
		cr.byGeoID[int(country.GeoId)] = country
	}
	return len(b), nil
}

func (cr *CountryRepository) Get(id int) (*rasp.TCountry, bool) {
	country, ok := cr.baseRepository.Get(int32(id))
	return country, ok
}

func (cr *CountryRepository) GetByGeoID(geoID int) (*rasp.TCountry, bool) {
	country, ok := cr.byGeoID[geoID]
	return country, ok
}

func (cr *CountryRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewCountryRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
