package references

import (
	"fmt"
	"io"
	"path/filepath"

	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath       string `config:"DICTS_RESOURCES_PATH" yaml:"resources_path"`
	UseDynamicResources bool   `yaml:"use_dynamic_resources"`
}

var DefaultConfig = Config{
	ResourcesPath:       "/app/data",
	UseDynamicResources: false,
}

type Registry struct {
	writerByPath         map[string]UpdatableWriter
	carriers             *CarrierRepository
	settlements          *SettlementsRepository
	stationCodes         *StationCodesRepository
	stations             *StationsRepository
	stationToSettlements *StationToSettlementsRepository
	regions              *RegionRepository
	countries            *CountryRepository
	config               Config
}

func (r *Registry) Settlements() *SettlementsRepository {
	return r.settlements
}

func (r *Registry) Stations() *StationsRepository {
	return r.stations
}

func (r *Registry) StationToSettlements() *StationToSettlementsRepository {
	return r.stationToSettlements
}

func (r *Registry) StationCodes() *StationCodesRepository {
	return r.stationCodes
}

func (r *Registry) Carriers() *CarrierRepository {
	return r.carriers
}

func (r *Registry) Regions() *RegionRepository {
	return r.regions
}

func (r *Registry) Countries() *CountryRepository {
	return r.countries
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func populateMany(path2writer map[string]UpdatableWriter) error {
	for path, writer := range path2writer {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("internal.references.registry:populateMany %w", err)
		}
		err = bytesIterator.Populate(writer)
		if err != nil {
			return fmt.Errorf("internal.references.registry:populateMany %w", err)
		}
	}
	return nil
}

func NewRegistry(c Config) (*Registry, error) {
	r := &Registry{
		carriers:             NewCarrierRepository(),
		settlements:          NewSettlementsRepository(),
		stationToSettlements: NewStationToSettlementsRepository(),
		stations:             NewStationsRepository(),
		stationCodes:         NewStationCodesRepository(),
		regions:              NewRegionRepository(),
		countries:            NewCountryRepository(),
		config:               c,
	}

	resourcePath, err := getResourcePath(c)
	if err != nil {
		return nil, fmt.Errorf("internal.references:NewRegistry %w", err)
	}

	r.updateWriterByPath(resourcePath)
	err = populateMany(r.writerByPath)
	if err != nil {
		return nil, fmt.Errorf("internal.references:NewRegistry %w", err)
	}
	return r, nil
}

func (r *Registry) OnUpdate() error {
	resourcePath, err := getResourcePath(r.config)
	if err != nil {
		return fmt.Errorf("internal.references.registry:OnUpdate %w", err)
	}
	r.updateWriterByPath(resourcePath)

	for path, writer := range r.writerByPath {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("internal.references.registry:OnUpdate %w", err)
		}
		if err := writer.UpdateFromSource(bytesIterator); err != nil {
			return fmt.Errorf("internal.references.registry:OnUpdate %w", err)
		}
	}
	return nil
}

func (r *Registry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/carrier.data":            r.carriers,
		resourcePath + "/settlement.data":         r.settlements,
		resourcePath + "/station_code.data":       r.stationCodes,
		resourcePath + "/station.data":            r.stations,
		resourcePath + "/settlement2station.data": r.stationToSettlements,
		resourcePath + "/region.data":             r.regions,
		resourcePath + "/country.data":            r.countries,
	}
}

func getResourcePath(config Config) (string, error) {
	resourcePath := config.ResourcesPath
	if config.UseDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(config.ResourcesPath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}
