package clients

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/library/go/httputil/headers"
	httputils "a.yandex-team.ru/travel/komod/trips/internal/helpers/http"
	"a.yandex-team.ru/travel/komod/trips/internal/services/activities/models"
)

type AfishaHTTPClient struct {
	config  AfishaConfig
	fetcher *httputils.Fetcher
}

func NewAfishaHTTPClient(config AfishaConfig, fetcher *httputils.Fetcher) *AfishaHTTPClient {
	return &AfishaHTTPClient{
		config:  config,
		fetcher: fetcher,
	}
}

func (c *AfishaHTTPClient) GetActivities(ctx context.Context, startDate time.Time, resultsLimit int, rawItemsLimit int, period int, geoID int) ([]models.AfishaEventInfo, error) {
	requestCtx, cancel := context.WithTimeout(ctx, c.config.RequestTimeout)
	defer cancel()
	request, err := c.buildGetAfishaBlockRequest(requestCtx, startDate, resultsLimit, rawItemsLimit, period, geoID)
	if err != nil {
		return nil, err
	}
	responseBytes, err := c.fetcher.Get(request)
	if err != nil {
		return nil, err
	}
	result := &models.AfishaPayload{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, err
	}
	return result.Events, nil
}

func (c *AfishaHTTPClient) buildGetAfishaBlockRequest(
	ctx context.Context,
	startDate time.Time,
	resultsLimit int,
	rawItemsLimit int,
	period int,
	geoID int,
) (*http.Request, error) {
	requestURL, _ := url.Parse(c.config.TravelAPIHost + "/api/trips/activities/v1/get_afisha_block")

	query := url.Values{}
	query.Set("startDate", startDate.Format(timeformats.RFC3339Date))
	query.Set("resultsLimit", fmt.Sprintf("%d", resultsLimit))
	query.Set("rawItemsLimit", fmt.Sprintf("%d", rawItemsLimit))
	query.Set("period", fmt.Sprintf("%d", period))
	query.Set("geoId", fmt.Sprintf("%d", geoID))
	requestURL.RawQuery = query.Encode()

	request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL.String(), nil)
	if err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := c.fetcher.SetServiceTicket(request); err != nil {
		return nil, err
	}
	return request, nil
}
