package clients

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"a.yandex-team.ru/library/go/httputil/headers"
	httputils "a.yandex-team.ru/travel/komod/trips/internal/helpers/http"
	"a.yandex-team.ru/travel/komod/trips/internal/services/activities/models"
)

type IziTravelHTTPClient struct {
	config  IziTravelConfig
	fetcher *httputils.Fetcher
}

func NewIziTravelHTTPClient(config IziTravelConfig, fetcher *httputils.Fetcher) *IziTravelHTTPClient {
	return &IziTravelHTTPClient{
		config:  config,
		fetcher: fetcher,
	}
}

func (c *IziTravelHTTPClient) GetActivities(ctx context.Context, resultsLimit int, geoID int) ([]models.IziTravelTourInfo, error) {
	requestCtx, cancel := context.WithTimeout(ctx, c.config.RequestTimeout)
	defer cancel()
	request, err := c.buildGetIziTravelBlockRequest(requestCtx, resultsLimit, geoID)
	if err != nil {
		return nil, err
	}
	responseBytes, err := c.fetcher.Get(request)
	if err != nil {
		return nil, err
	}
	result := &models.IziTravelPayload{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, err
	}
	return result.Tours, nil
}

func (c *IziTravelHTTPClient) buildGetIziTravelBlockRequest(
	ctx context.Context,
	resultsLimit int,
	geoID int,
) (*http.Request, error) {
	requestURL, _ := url.Parse(c.config.TravelAPIHost + "/api/trips/activities/v1/get_izi_travel_block")

	query := url.Values{}
	query.Set("resultsLimit", fmt.Sprintf("%d", resultsLimit))
	query.Set("geoId", fmt.Sprintf("%d", geoID))
	requestURL.RawQuery = query.Encode()

	request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL.String(), nil)
	if err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := c.fetcher.SetServiceTicket(request); err != nil {
		return nil, err
	}
	return request, nil
}
