package contentadmin

import (
	"fmt"

	"github.com/cenkalti/backoff/v4"
	"github.com/opentracing/opentracing-go"
	"golang.org/x/net/context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/contentadmin"
	v1 "a.yandex-team.ru/travel/marketing/content/v1"
)

type RetryableClient struct {
	config             Config
	backoffPolicy      backoff.BackOff
	contentAdminClient *contentadmin.ContentAdminClient
}

func NewRetryableClient(logger log.Logger, selfTvmID uint32, config Config) (Client, error) {
	contentAdminClient := contentadmin.NewContentAdminClient(logger, selfTvmID, config.toLibConfig())
	if contentAdminClient == nil {
		return nil, fmt.Errorf("unable to create content admin client")
	}
	return &RetryableClient{
		config:             config,
		backoffPolicy:      getBackOffPolicy(),
		contentAdminClient: contentAdminClient,
	}, nil
}

func (c *RetryableClient) GetCityImageByID(ctx context.Context, cityID int) (cityImage *v1.CityImage, apiErr error) {
	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, "services.contentadmin.RetryableClient.GetCityImageByID")
	defer tracingSpan.
		SetTag("cityID", cityID).
		Finish()

	var rsp *v1.CityImage
	err := backoff.Retry(
		func() (err error) {
			rsp, err = c.contentAdminClient.GetCityImageByID(ctx, cityID)
			return err
		},
		backoff.WithContext(c.backoffPolicy, ctx),
	)
	return rsp, err
}
