package processor

import (
	"context"
	"time"

	"github.com/cenkalti/backoff/v4"
	"google.golang.org/grpc"

	"a.yandex-team.ru/travel/komod/trips/api/processor/v1"
)

type Service struct {
	config Config
	client processor.ProcessorServiceV1Client
	ticker *time.Ticker
}

func NewService(connection *grpc.ClientConn, config Config) *Service {
	grpcClient := processor.NewProcessorServiceV1Client(connection)
	return &Service{
		config: config,
		client: grpcClient,
		ticker: time.NewTicker(getRequestInterval(config.RPSLimit)),
	}
}

func (s Service) Process(ctx context.Context, req *processor.ProcessTripEntityReq) error {
	<-s.ticker.C
	return backoff.Retry(
		func() error {
			ctx, cancel := context.WithTimeout(ctx, s.config.Timeout)
			defer cancel()

			_, err := s.client.ProcessTripEntity(ctx, req)
			return err
		},
		backoff.WithContext(backoff.NewConstantBackOff(time.Second), ctx),
	)
}

func getRequestInterval(rpsLimit float32) time.Duration {
	return time.Second * 1000 / time.Duration(rpsLimit*1000)
}
