package unprocessedorders

import (
	"context"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/jonboulle/clockwork"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/komod/trips/api/processor/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type LogbrokerProducer interface {
	Write(msg proto.Message) error
	Stat() persqueue.WriterStat
	Close() error
}

type Config struct {
	Topic      string `config:"UNPROCESSED_ORDERS_TOPIC" yaml:"topic"`
	ProducerID string `config:"UNPROCESSED_ORDERS_PRODUCER_ID" yaml:"producer_id"`
	Endpoint   string `config:"LOGBROKER_PRODUCER_ENDPOINT" yaml:"endpoint"`
	Token      string `config:"LOGBROKER_PRODUCER_TOKEN,required" yaml:"token"`
	Mock       bool   `config:"MOCK_UNPROCESSED_ORDERS" yaml:"mock"`
}

var DefaultConfig = Config{
	Topic:      "/avia/development/trips/unprocessed-orders",
	Endpoint:   "logbroker.yandex.net",
	ProducerID: "unprocessed-orders",
	Mock:       true,
}

type Storage interface {
	Count(ctx context.Context) (int, error)
	Upsert(context.Context, *UnprocessedOrder) (bool, error)
}

type Service struct {
	logger log.Logger

	logbrokerProducer LogbrokerProducer
	storage           Storage
	clock             clockwork.Clock
}

func NewService(logger log.Logger, logbrokerProducer LogbrokerProducer, clock clockwork.Clock, storage Storage) *Service {
	go sendProducerStat(logbrokerProducer)
	return &Service{logger: logger, logbrokerProducer: logbrokerProducer, clock: clock, storage: storage}
}

func (s *Service) Put(ctx context.Context, req *processor.ProcessTripEntityReq) error {
	ctx = s.fillCtx(ctx, req)
	ctxlog.Info(ctx, s.logger, "putting unprocessed order")
	err := s.logbrokerProducer.Write(req)
	if err != nil {
		sendEventMetric(map[string]string{"status": "error", "phase": "write_to_logbroker", "order_type": orderTypeTravel})
		ctxlog.Error(ctx, s.logger, "failed to write proto-message into logbroker", log.Error(err))
		return err
	} else {
		sendEventMetric(map[string]string{"status": "success", "order_type": orderTypeTravel})
		ctxlog.Info(
			ctx,
			s.logger,
			"unprocessed order has been sent to logbroker successfully",
		)
	}
	return nil
}

func (s *Service) StoreFailed(ctx context.Context, message *processor.ProcessTripEntityReq) error {
	orderID := message.GetOrder().GetId()
	inserted, err := s.storage.Upsert(ctx, &UnprocessedOrder{ID: orderID})
	if err != nil {
		return nil
	}
	if inserted {
		metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, nil, storedFailedOrders).Inc()
	}
	return nil
}

func (s *Service) MonitorFailedOrdersCount() {
	for range time.Tick(1 * time.Minute) {
		ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		count, err := s.storage.Count(ctx)
		cancel()
		if err != nil {
			s.logger.Error("failed to count failed unprocessed orders")
		} else {
			metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, totalFailed).Set(float64(count))
		}
	}
}

func (s *Service) PutOrderID(ctx context.Context, orderID orders.ID, retriesCount uint32) error {
	req := &processor.ProcessTripEntityReq{
		CollectedAt: timestamppb.Now(),
		Entity: &processor.ProcessTripEntityReq_Order{Order: &processor.OrderEntity{
			Id: string(orderID),
		}},
		RetriesLeft: retriesCount,
	}
	return s.Put(ctx, req)
}

func (s *Service) fillCtx(ctx context.Context, req *processor.ProcessTripEntityReq) context.Context {
	orderID := req.GetOrder().GetId()
	retriesLeft := req.GetRetriesLeft()
	collectedAt := req.GetCollectedAt().AsTime()
	return ctxlog.WithFields(
		ctx,
		log.String("orderID", orderID),
		log.Time("collectedAt", collectedAt),
		log.UInt32("retriesLeft", retriesLeft),
	)
}

func sendProducerStat(producer LogbrokerProducer) {
	t := time.NewTicker(5 * time.Second)
	defer t.Stop()
	for range t.C {
		stat := producer.Stat()
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "inflight").Set(float64(stat.Inflight))
		metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, nil, "mem_usage").Set(float64(stat.MemUsage))
	}
}

func sendEventMetric(tags map[string]string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(metricsPrefix, tags, sentMessagesMetricName).Inc()
}
