package weather

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/library/go/httputil/headers"
	httputils "a.yandex-team.ru/travel/komod/trips/internal/helpers/http"
)

type HTTPClient struct {
	config  HTTPClientConfig
	fetcher *httputils.Fetcher
}

func NewHTTPClient(config HTTPClientConfig, fetcher *httputils.Fetcher) *HTTPClient {
	return &HTTPClient{
		config:  config,
		fetcher: fetcher,
	}
}

func (c *HTTPClient) GetByGeoID(ctx context.Context, geoID int, startDate time.Time, days int) (*GetByGeoIDResponse, error) {
	ctx, cancel := context.WithTimeout(ctx, c.config.RequestTimeout)
	defer cancel()
	request, err := c.buildGetByGeoIDRequest(ctx, startDate, days, geoID)
	if err != nil {
		return nil, err
	}
	responseBytes, err := c.fetcher.Get(request)
	if err != nil {
		return nil, err
	}
	result := &GetByGeoIDResponse{}
	if err = json.Unmarshal(responseBytes, &result); err != nil {
		return nil, err
	}
	return result, nil
}

func (c *HTTPClient) buildGetByGeoIDRequest(
	ctx context.Context,
	startDate time.Time,
	days int,
	geoID int,
) (*http.Request, error) {
	requestURL, _ := url.Parse(c.config.TravelAPIHost + "/api/weather/v1/get_by_geo_id")

	query := url.Values{}
	query.Set("startDate", startDate.Format(timeformats.RFC3339Date))
	query.Set("days", fmt.Sprintf("%d", days))
	query.Set("geoId", fmt.Sprintf("%d", geoID))
	requestURL.RawQuery = query.Encode()

	request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL.String(), nil)
	if err != nil {
		return nil, err
	}
	request.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := c.fetcher.SetServiceTicket(request); err != nil {
		return nil, err
	}
	return request, nil
}
