package sharedflights

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"
	"github.com/cenkalti/backoff/v4"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
)

const (
	flightOnDateMethod = "flight"
)

var ErrFlightNotFound = fmt.Errorf("flight not found")

type Client struct {
	httpClient *http.Client
	config     Config
	logger     log.Logger
}

func NewClient(
	config Config,
	httpClient *http.Client,
	logger log.Logger,
) *Client {
	return &Client{
		httpClient: httpClient,
		config:     config,
		logger:     logger,
	}
}

func (c *Client) GetFlightInfo(ctx context.Context, flightNumber string, departureDate time.Time) (Flight, error) {
	handlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "shared-flights request")
	defer handlerSpan.Finish()

	requestURL := fmt.Sprintf("%s/%s", c.config.BaseURL, buildFlightRequestURL(flightNumber, departureDate))
	flight := Flight{}
	requestFunc := func() error {
		ctx, cancel := context.WithTimeout(ctx, c.config.FlightOnDateTimeout)
		defer cancel()
		request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL, nil)
		if err != nil {
			return err
		}

		response, err := c.getResponse(request)
		if response != nil && response.Body != nil {
			defer response.Body.Close()
		}
		if err != nil {
			return err
		}

		if err := json.NewDecoder(response.Body).Decode(&flight); err != nil {
			ctxlog.Error(ctx, c.logger, "got invalid flight format from shared-flights", log.Error(err))
		}
		return nil
	}
	if err := c.retryRequest(requestFunc); err != nil {
		return flight, err
	}
	return flight, nil
}

func buildFlightRequestURL(flightNumber string, departureDate time.Time) string {
	values := url.Values{}
	values.Set("service", "trips")
	flightNumberParts := strings.Split(flightNumber, " ")
	companyIata := flightNumberParts[0]
	number := flightNumberParts[1]
	return fmt.Sprintf(
		"api/v1/%s/%s/%s/%s/?%s",
		flightOnDateMethod,
		companyIata,
		number,
		formatDate(departureDate),
		values.Encode(),
	)
}

func (c *Client) getResponse(request *http.Request) (*http.Response, error) {
	response, err := c.httpClient.Do(request)
	if err != nil {
		if !errors.Is(err, context.DeadlineExceeded) && !errors.Is(err, context.Canceled) {
			ctxlog.Error(request.Context(), c.logger, "an error occurred while getting models", log.Error(err))
		}
		return nil, err
	}
	if response.StatusCode != http.StatusOK {
		if response.StatusCode == http.StatusNotFound {
			return nil, backoff.Permanent(ErrFlightNotFound)
		}
		err = fmt.Errorf("shared-flights has responded with code %d", response.StatusCode)
		ctxlog.Error(request.Context(), c.logger, err.Error())
		if response.StatusCode >= http.StatusBadRequest && response.StatusCode < http.StatusInternalServerError {
			return response, backoff.Permanent(err)
		}
		return response, err
	}
	return response, nil
}

func (c *Client) retryRequest(request func() error) error {
	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     20 * time.Millisecond,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         200 * time.Millisecond,
		MaxElapsedTime:      2 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}

	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}

func formatDate(date time.Time) string {
	return date.Format(timeformats.RFC3339Date)
}
