package span

import (
	"sort"
	"time"

	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
)

type Helper struct {
	pointComparator *point.Comparator
	spanComparator  *SpanComparator
}

func NewHelper(pointComparator *point.Comparator, spanComparator *SpanComparator) *Helper {
	return &Helper{pointComparator: pointComparator, spanComparator: spanComparator}
}

func (h *Helper) RemoveDuplicatedVisits(visits []models.Visit) []models.Visit {
	sort.Stable(visitsByTime(visits))
	if len(visits) == 0 {
		return nil
	}

	var result []models.Visit
	var addedPoints []models.Point
	for _, v := range visits {
		alreadyAdded := false
		for _, addedPoint := range addedPoints {
			alreadyAdded = alreadyAdded || h.pointComparator.SamePoints(addedPoint, v.Point())
		}
		if !alreadyAdded {
			result = append(result, v)
			addedPoints = append(addedPoints, v.Point())
		}
	}
	return result
}

type visitsByTime []models.Visit

func (b visitsByTime) Len() int {
	return len(b)
}

func (b visitsByTime) Less(i, j int) bool {
	return b[i].When().Before(b[j].When())
}

func (b visitsByTime) Swap(i, j int) {
	b[i], b[j] = b[j], b[i]
}

func (h *Helper) ExtractVisitsRemovingExtremes(spans []models.Span) []models.Visit {
	spans = filterNilSpans(spans)

	if len(spans) == 0 {
		return nil
	}
	h.SortByStart(spans)

	lastSpan := spans[len(spans)-1]
	fromHome := spans[0].IsTransport()
	isRoundTrip := fromHome && lastSpan.IsTransport() && h.pointComparator.SamePoints(spans[0].Start().Point(), lastSpan.End().Point())
	visits := ExtractVisitsFromSpans(spans)

	left := 0
	if fromHome {
		left++
	}

	right := len(visits)
	if isRoundTrip {
		right--
	}

	if left >= right {
		return []models.Visit{lastSpan.End()}
	}

	return visits[left:right]
}

func (h *Helper) allowedTransferDuration(duration time.Duration) bool {
	return duration < units.Day
}

func (h *Helper) ReduceTransfers(spans []models.Span) []models.Span {
	spans = filterNilSpans(spans)

	if len(spans) < 2 {
		return spans
	}
	h.SortByStart(spans)

	var result = []models.Span{spans[0]}
	for i := 1; i < len(spans); i++ {
		left, right := spans[i-1], spans[i]

		d := right.Start().When().Sub(left.End().When())
		if !left.IsTransport() || !right.IsTransport() || !h.allowedTransferDuration(helpers.AbsDuration(d)) {
			result = append(result, right)
			continue
		}

		lastIndex := len(result) - 1
		result[lastIndex] = models.NewSpan(left.Start(), right.End(), left.IsTransport() || right.IsTransport())
	}
	if len(result) == 1 && h.pointComparator.SamePoints(result[0].Start().Point(), result[0].End().Point()) {
		return spans
	}
	return result
}

func (h Helper) SortByStart(spans []models.Span) {
	if sort.IsSorted(spanByStartTime(spans)) {
		return
	}
	sort.Sort(spanByStartTime(spans))
}

type spanByStartTime []models.Span

func (b spanByStartTime) Len() int {
	return len(b)
}

func (b spanByStartTime) Less(i, j int) bool {
	return b[i].Start().When().Before(b[j].Start().When())
}

func (b spanByStartTime) Swap(i, j int) {
	b[i], b[j] = b[j], b[i]
}

func (h *Helper) IsRoundTrip(spans ...models.Span) bool {
	spans = filterNilSpans(spans)

	if len(spans) == 0 {
		return false
	}
	left := h.GetLeftMostSpan(spans...)
	right := h.GetRightMostSpan(spans...)
	return h.pointComparator.SamePoints(left.Start().Point(), right.End().Point())
}

func (h Helper) GetLeftMostSpan(spans ...models.Span) models.Span {
	spans = filterNilSpans(spans)
	if len(spans) == 0 {
		return nil
	}
	left := spans[0]
	for _, s := range spans {
		if s.Start().When().Before(left.Start().When()) {
			left = s
		}
	}
	return left
}

func filterNilSpans(spans []models.Span) (result []models.Span) {
	for _, s := range spans {
		if s != nil {
			result = append(result, s)
		}
	}
	return result
}

func (h Helper) GetRightMostSpan(spans ...models.Span) models.Span {
	spans = filterNilSpans(spans)
	if len(spans) == 0 {
		return nil
	}
	right := spans[0]
	for _, s := range spans {
		if s.End().When().After(right.End().When()) {
			right = s
		}
	}
	return right
}

func (h *Helper) MakeGaps(spans []models.Span) (result []models.Span) {
	spans = filterNilSpans(spans)
	if len(spans) == 0 {
		return nil
	}

	visits := make(visitsFromSpan, 0, len(spans)*2)
	for _, s := range spans {
		visits = append(visits, visitFromSpan{
			visit:       s.Start(),
			startOfSpan: true,
		})
		visits = append(visits, visitFromSpan{
			visit:       s.End(),
			startOfSpan: false,
		})
	}
	sort.Sort(visits)

	prev := visits[0]
	spanStackCount := 1
	for _, current := range visits[1:] {
		if spanStackCount == 0 {
			result = append(result, models.NewSpan(prev.visit, current.visit, false))
		}

		prev = current
		if current.startOfSpan {
			spanStackCount++
		} else {
			spanStackCount--
		}
	}
	return result
}

func (h *Helper) RemovePreviousSpans(spans []models.Span) (result []models.Span) {
	now := time.Now()
	for _, span := range spans {
		if span.End().When().After(now) {
			result = append(result, span)
		}
	}
	return result
}

func (h *Helper) GetTransportSpans(spans []models.Span) (result []models.Span) {
	for _, span := range spans {
		if span.IsTransport() {
			result = append(result, span)
		}
	}
	return result
}

func (h *Helper) GetLocationSpans(spans []models.Span) (result []models.Span) {
	for _, span := range spans {
		if !span.IsTransport() {
			result = append(result, span)
		}
	}
	return result
}

type visitFromSpan struct {
	visit       models.Visit
	startOfSpan bool
}

type visitsFromSpan []visitFromSpan

func (b visitsFromSpan) Len() int {
	return len(b)
}

func (b visitsFromSpan) Less(i, j int) bool {
	if b[i].visit.When().Equal(b[j].visit.When()) {
		return b[i].startOfSpan && !b[j].startOfSpan
	}
	return b[i].visit.When().Before(b[j].visit.When())
}

func (b visitsFromSpan) Swap(i, j int) {
	b[i], b[j] = b[j], b[i]
}
