package matcher

import (
	"a.yandex-team.ru/library/go/units"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
)

type PointComparator interface {
	SamePoints(models.Point, models.Point) bool
	InSameSubject(models.Point, models.Point) bool
	InSameCountry(models.Point, models.Point) bool
}

type SpanHelper interface {
	IsRoundTrip(spans ...models.Span) bool
}

type RuleFactory struct {
	comparator PointComparator
	spanHelper SpanHelper
}

const popularTripDuration = 14 * units.Day

func NewRuleFactory(comparator PointComparator, spanHelper SpanHelper) *RuleFactory {
	return &RuleFactory{
		comparator: comparator,
		spanHelper: spanHelper,
	}
}

func (f RuleFactory) Make() Rule {
	return NewOrRule(
		NewPredicateRule(f.IsCorrectSpansInterposition),
		NewWholeTripRule(NewOrRule(
			NewPredicateRule(f.IsLongTransfer),
			NewPredicateRule(f.IsRoundTrip),
			NewPredicateRule(f.InSameRegion),
			//NewPredicateRule(f.InSameCountry),
		)),
	)
}

func (f RuleFactory) IsCorrectSpansInterposition(lhs, rhs models.Span) bool {
	if span.IsSuperSpan(lhs, rhs) || span.IsSuperSpan(rhs, lhs) {
		// at least one pair of visits should match
		return f.comparator.InSameSubject(lhs.Start().Point(), rhs.Start().Point()) ||
			f.comparator.InSameSubject(rhs.End().Point(), lhs.End().Point())
	}
	lhs, rhs = orderByStart(lhs, rhs)
	gapLessThanDay := rhs.Start().When().Sub(lhs.End().When()) <= units.Day
	if span.DatesIntersect(lhs, rhs) {
		for _, visits := range getVisitCombinations(lhs, rhs) {
			if f.comparator.InSameSubject(visits[0].Point(), visits[1].Point()) {
				return true
			}
		}
	} else if gapLessThanDay {
		return f.comparator.InSameSubject(lhs.End().Point(), rhs.Start().Point())
	}
	return false
}

func (f RuleFactory) InSameRegion(lhs, rhs models.Span) bool {
	lhs, rhs = orderByStart(lhs, rhs)
	return f.comparator.InSameSubject(lhs.End().Point(), rhs.Start().Point())
}

func (f RuleFactory) InSameCountry(lhs, rhs models.Span) bool {
	lhs, rhs = orderByStart(lhs, rhs)
	return f.comparator.InSameCountry(lhs.End().Point(), rhs.Start().Point())
}

func getVisitCombinations(lhs models.Span, rhs models.Span) [][]models.Visit {
	return [][]models.Visit{
		{lhs.Start(), rhs.Start()},
		{lhs.Start(), rhs.End()},
		{lhs.End(), rhs.Start()},
		{lhs.End(), rhs.End()},
	}
}

func (f RuleFactory) IsLongTransfer(lhs, rhs models.Span) bool {
	lhs, rhs = orderByStart(lhs, rhs)
	samePoints := f.comparator.SamePoints(lhs.End().Point(), rhs.Start().Point())
	isLongTransfer := rhs.Start().When().Sub(lhs.End().When()) >= units.Day
	return samePoints && isLongTransfer
}

func (f RuleFactory) IsRoundTrip(lhs, rhs models.Span) bool {
	return f.spanHelper.IsRoundTrip(lhs, rhs)
}

func isLongerTripThanPopular(lhs, rhs models.Span) bool {
	leftTime := lhs.Start().When()
	if rhs.Start().When().Before(leftTime) {
		leftTime = rhs.Start().When()
	}

	rightTime := lhs.End().When()
	if rhs.End().When().After(rightTime) {
		rightTime = rhs.End().When()
	}
	return rightTime.Sub(leftTime) > popularTripDuration
}

func orderByStart(lhs, rhs models.Span) (models.Span, models.Span) {
	if lhs.Start().When().After(rhs.Start().When()) {
		lhs, rhs = rhs, lhs
	}
	return lhs, rhs
}
