package models

import (
	"time"

	tripsapi "a.yandex-team.ru/travel/komod/trips/api/trips/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	ordercommons "a.yandex-team.ru/travel/orders/proto"
)

type Trips []*Trip

type Trip struct {
	ID         string
	PassportID string
	OrderInfos map[orders.ID]OrderInfo
}

func NewTrip(id, passportID string) *Trip {
	return &Trip{
		ID:         id,
		PassportID: passportID,
		OrderInfos: make(map[orders.ID]OrderInfo),
	}
}

func (t Trip) GetOrderIDs() (ids []orders.ID) {
	for id := range t.OrderInfos {
		ids = append(ids, id)
	}
	return ids
}

func (t Trip) State() tripsapi.TripState {
	for _, orderInfo := range t.OrderInfos {
		if !orderInfo.CancelledByUser() {
			return tripsapi.TripState_TRIP_STATE_CONFIRMED
		}
	}
	return tripsapi.TripState_TRIP_STATE_CANCELLED
}

func (t Trip) Cancelled() bool {
	return t.State() == tripsapi.TripState_TRIP_STATE_CANCELLED
}

func (t Trip) GetActiveSpans() (result []models.Span) {
	for _, orderInfo := range t.OrderInfos {
		if orderInfo.CancelledByUser() {
			continue
		}
		result = append(result, orderInfo.Spans...)
	}
	return result
}

func (t Trip) GetSpans() (result []models.Span) {
	for _, orderInfo := range t.OrderInfos {
		result = append(result, orderInfo.Spans...)
	}
	return result
}

func (t Trip) GetActiveOrAllSpans() (result []models.Span) {
	result = t.GetActiveSpans()
	if len(result) > 0 {
		return result
	}
	return t.GetSpans()
}

func (t *Trip) UpsertOrder(order OrderInfo) {
	t.OrderInfos[order.ID] = order
}

func (t Trip) StartTime() time.Time {
	return span.GetStartTime(t.GetSpans())
}

func (t Trip) EndTime() time.Time {
	return span.GetEndTime(t.GetSpans())
}

func (t Trip) StartVisit() models.Visit {
	return span.GetStartVisit(t.GetSpans())
}

func (t Trip) EndVisit() models.Visit {
	return span.GetEndVisit(t.GetSpans())
}

type OrderInfo struct {
	ID    orders.ID
	State ordercommons.EDisplayOrderState
	Spans []models.Span
}

func (o *OrderInfo) CancelledByUser() bool {
	return o.State == ordercommons.EDisplayOrderState_OS_REFUNDED
}

func NewOrderInfo(ID orders.ID, state ordercommons.EDisplayOrderState, spans []models.Span) OrderInfo {
	return OrderInfo{ID: ID, State: state, Spans: spans}
}
