#!/usr/bin/env python3
# coding: utf-8

import os
import shutil

from utils import make_deploy_check_sender


FREE_SPACE_PERCENT = float(os.getenv('DISK_FREE_CHECK_PERCENT', '10'))
ROOTS = os.getenv('DISK_FREE_CHECK_ROOTS', '/,/ephemeral').split(',')

CHECK_NAME = 'disk-free-check'


deploy_check_sender = make_deploy_check_sender(CHECK_NAME, tags=['disk'])


def execute_free_space_check():
    for fs_root in ROOTS:
        try:
            if not _root_has_enough_free_space(fs_root):
                return 'Filesystem {} is running out of free space'.format(fs_root)
        except FileNotFoundError:
            pass
    return None


def _root_has_enough_free_space(fs_root):
    usage = _get_free_space(fs_root)
    free_percent = usage.free / usage.total * 100
    if free_percent < FREE_SPACE_PERCENT:
        return False
    else:
        return True


def _get_free_space(fs_root):
    return shutil.disk_usage(fs_root)


if __name__ == '__main__':
    if os.getenv('SKIP_DICK_FREE_CHECK'):
        deploy_check_sender.send_event('OK', 'OK')
        exit(0)
    error_message = execute_free_space_check()
    if error_message:
        deploy_check_sender.send_event('CRIT', error_message)
    else:
        deploy_check_sender.send_event('OK', 'OK')
