#!/usr/bin/env python3
# coding: utf8
import argparse
import logging.config
import os

from collections import Counter

from aiohttp.web import Application, Request, run_app, post, get, json_response

logging.config.dictConfig({
    'version': 1,
    'formatters': {
        'qloud': {
            'class': 'ylog.QloudJsonFormatter',
        }
    },
    'handlers': {
        'stream': {
            'class': 'logging.StreamHandler',
            'level': 'INFO',
            'formatter': 'qloud'
        },
    },
    'loggers': {
        'application': {
            'handlers': ['stream']
        }
    },
})
logger = logging.getLogger('application')


class MetricsStorage:
    def __init__(self, override_tags=False):
        self.metrics_accumulator = {}
        self.override_tags = override_tags

    def add_application(self, name):
        self.metrics_accumulator[name] = {}

    def add_histogram(self, name, value):
        for accumulator in self.metrics_accumulator.values():
            if name not in accumulator:
                accumulator[name] = Counter()
            accumulator[name][value[0]] += value[1]

    def add_metric(self, name, value):
        for accumulator in self.metrics_accumulator.values():
            accumulator[name] = value

    def add(self, entry):
        tags = entry.get('tags', {})
        tags_prefix = ';'.join('{}={}'.format(tag, value) for tag, value in tags.items())
        metrics = [{'name': entry['name'], 'val': entry['val']}] if 'values' not in entry else entry['values']
        for metric in metrics:
            name = metric['name']
            value = metric['val']
            if self.override_tags:
                name = '{};{}'.format(tags_prefix, name)
            if name.endswith('_ahhh'):
                for bucket in value:
                    self.add_histogram(name, bucket)
            else:
                self.add_metric(name, value)

    def dump(self, application):
        result = []
        for name, value in self.metrics_accumulator[application].items():
            if name.endswith('_ahhh'):
                result.append([name, sorted(value.items())])
            else:
                result.append([name, value])

        self.metrics_accumulator[application] = {}
        return result


class App(Application):
    def __init__(self, override_tags, *args, **kwargs):
        super(App, self).__init__(*args, **kwargs)
        self.storage = MetricsStorage(override_tags)
        self.override_tags = override_tags

    async def add_metrics(self, request: Request):
        """
        Возможное содержимое запросов:
        https://doc.yandex-team.ru/Search/golovan-quickstart/concepts/push.html

            для одиночных метрик
            [{
                "name": "foo_xxxx",
                "tags": {
                    "itype": "someitype",
                    "ctype": "ctype10",
                    "_priority": 22
                },
                "ttl": 30,
                "val": 3
            }]

            для множественных метрик
            [{
                "tags": {
                    "itype": "someitype",
                    "ctype": "ctype10",
                    "_priority": 22
                },
                "ttl": 30,
                "values": [{
                    "name": "sig1_xxxx",
                    "val": 1
                }, {
                    "name": "sig2_nnnn",
                    "val": 2
                }]
            }]
        :param request: Request
        :return:
        """
        try:
            for entry in await request.json():
                self.storage.add(entry)
        except Exception as ex:
            logger.exception('Exception while sending metrics happens')
            return json_response({
                'error': repr(ex),
                'error_code': repr(ex)
            }, status=400)
        return json_response({'status': 'ok'})

    def get_metrics_factory(self, application_name):
        self.storage.add_application(application_name)

        async def get_metrics(request: Request):
            """
            https://wiki.yandex-team.ru/golovan/stat-handle/?from=%252Fjandekspoisk%252Fsepe%252Fmonitoring%252Fstat-handle%252F
            :param request: Request
            :return:
            """
            return json_response(self.storage.dump(application_name))
        return get_metrics


def run():
    parser = argparse.ArgumentParser()
    parser.add_argument('--override-tags', default=os.environ.get('UNISTAT_OVERRIDE_TAGS', False))
    parser.add_argument('--host', default=os.environ.get('UNISTAT_HOST', '127.0.0.1,::1'))
    parser.add_argument('--port', default=os.environ.get('UNISTAT_PORT', 11005))
    args = parser.parse_args()
    app = App(args.override_tags)
    app.add_routes([
        post("/", app.add_metrics),
        post("/metrics", app.add_metrics),
        get("/unistat", app.get_metrics_factory('golovan')),
        get("/unistat/solomon", app.get_metrics_factory('solomon')),
    ])
    hosts = [h.strip() for h in args.host.split(',')]
    run_app(app, host=hosts, port=args.port)


if __name__ == '__main__':
    run()
