package yav

import (
	"context"
	"github.com/heetch/confita/backend"
	"os"
	"strings"
)

// Backend loads keys from Yav
type Backend struct {
	resolver SecretsResolver
	secUID   string
}

type SecUIDConfig struct {
	Production  string
	Testing     string
	Development string
}

type SecretsResolver interface {
	GetSecretValue(secUID string, key string) (string, error)
}

// NewBackend creates a configuration loader that loads from Yav
func NewBackend(resolver SecretsResolver, yavConfig SecUIDConfig) *Backend {
	var secUID string

	envType := os.Getenv("YENV_TYPE")
	switch envType {
	case "production":
		secUID = yavConfig.Production
	case "development":
		secUID = yavConfig.Development
	case "testing":
		secUID = yavConfig.Testing
	}

	return &Backend{
		resolver: resolver,
		secUID:   secUID,
	}
}

// Get loads the given key from YAV
// If the key is not found, this backend tries again by turning any kebabcase key to snakecase and
// lowercase letters to uppercase.
func (b *Backend) Get(ctx context.Context, key string) ([]byte, error) {
	if b.secUID == "" {
		return nil, backend.ErrNotFound
	}

	val, err := b.resolver.GetSecretValue(b.secUID, key)
	if err == nil && val != "" {
		return []byte(val), nil
	}

	upperKey := strings.Replace(strings.ToUpper(key), "-", "_", -1)
	if upperKey == key {
		return nil, backend.ErrNotFound
	}

	val, err = b.resolver.GetSecretValue(b.secUID, upperKey)
	if err == nil && val != "" {
		return []byte(val), nil
	}

	return nil, backend.ErrNotFound
}

// Name returns the name of the backend.
func (b *Backend) Name() string {
	return "yav"
}
