package configuration

import (
	"os"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"
	"github.com/heetch/confita/backend/flags"

	"a.yandex-team.ru/travel/library/go/configuration/backends/yav"
)

const (
	TestingConfigPath     = "/app/config.testing.yaml"
	ProductionConfigPath  = "/app/config.production.yaml"
	DevelopmentConfigPath = "/app/config.development.yaml"
	UnstableConfigPath    = "/app/config.unstable.yaml"
)

type LoaderOption func([]backend.Backend) []backend.Backend

func exists(path string) bool {
	if _, err := os.Stat(path); err != nil && os.IsNotExist(err) {
		return false
	}
	return true
}

func getPathToConfig() string {
	path, exists := os.LookupEnv("CONFIG_PATH")
	if exists {
		return path
	}
	envType := os.Getenv("YENV_TYPE")
	switch envType {
	case "production":
		path = ProductionConfigPath
	case "development":
		path = DevelopmentConfigPath
	case "testing":
		path = TestingConfigPath
	case "unstable":
		path = UnstableConfigPath
	}
	return path
}

func NewDefaultConfitaLoader() *confita.Loader {
	pathToConfig := getPathToConfig()
	return NewConfitaLoaderWithYamlPath(pathToConfig)
}

func NewConfitaLoaderWithYamlPath(path string) *confita.Loader {
	var loader *confita.Loader
	if path != "" && exists(path) {
		loader = confita.NewLoader(
			file.NewBackend(path),
			env.NewBackend(),
			flags.NewBackend(),
		)
	} else {
		loader = confita.NewLoader(
			env.NewBackend(),
			flags.NewBackend(),
		)
	}
	return loader
}

func NewConfitaLoader(options ...LoaderOption) *confita.Loader {
	backends := []backend.Backend{
		env.NewBackend(),
		flags.NewBackend(),
	}

	for _, opt := range options {
		backends = opt(backends)
	}

	return confita.NewLoader(backends...)
}

func WithYavBackend(resolver yav.SecretsResolver, secUIDConfig yav.SecUIDConfig) LoaderOption {
	return func(backends []backend.Backend) []backend.Backend {
		return append(backends, yav.NewBackend(resolver, secUIDConfig))
	}
}

func WithYamlBackend(path string) LoaderOption {
	return func(backends []backend.Backend) []backend.Backend {
		if path != "" && exists(path) {
			return append([]backend.Backend{file.NewBackend(path)}, backends...)
		}

		return backends
	}
}

func WithYamlDefaultBackend() LoaderOption {
	pathToConfig := getPathToConfig()
	return WithYamlBackend(pathToConfig)
}
