package contentadmin

import (
	"context"
	"fmt"
	"time"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	balancerconfig "a.yandex-team.ru/travel/library/go/grpcutil/client/balancer_config"
	"a.yandex-team.ru/travel/library/go/tvm"
	v1 "a.yandex-team.ru/travel/marketing/content/v1"
)

type ContentAdminConfig struct {
	TvmID   uint32
	Host    string
	Timeout time.Duration
}

type ContentAdminClient struct {
	tvmInterceptor grpc.UnaryClientInterceptor
	config         ContentAdminConfig
	interceptors   []grpc.UnaryClientInterceptor
}

func NewContentAdminClient(
	logger log.Logger,
	selfTvmID uint32,
	config ContentAdminConfig,
	options ...Option,
) *ContentAdminClient {
	tvmHelper := tvm.NewDeployTvmHelper(logger, &tvm.TvmHelperConfig{
		SelfID:    selfTvmID,
		WhiteList: []uint32{},
	})
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(config.TvmID)
	if tvmInterceptor == nil {
		return nil
	}

	client := &ContentAdminClient{tvmInterceptor, config, []grpc.UnaryClientInterceptor{}}
	for _, opt := range options {
		opt(client)
	}
	return client
}

func (c *ContentAdminClient) GetCityImageByID(ctx context.Context, cityID int) (cityImage *v1.CityImage, apiErr error) {
	response, err := c.sendRequest(ctx, cityID)
	if err != nil {
		apiErr = err
		return
	}

	if response.ImageUrl == "" {
		apiErr = fmt.Errorf("content-admin client has returned an empty image link")
		return
	}

	return response, nil
}

func (c *ContentAdminClient) sendRequest(ctx context.Context, cityID int) (cityImage *v1.CityImage, apiErr error) {
	ctx, cancel := context.WithTimeout(ctx, c.config.Timeout)
	defer cancel()
	interceptors := c.interceptors
	interceptors = append(interceptors, c.tvmInterceptor)
	dialOpts := []grpc.DialOption{
		grpc.WithDefaultServiceConfig(balancerconfig.RoundRobin()),
		grpc.WithInsecure(),
		grpc.WithBlock(),
		grpc.WithChainUnaryInterceptor(interceptors...),
	}
	gConn, err := grpc.DialContext(
		ctx,
		c.config.Host,
		dialOpts...,
	)
	if err != nil {
		apiErr = fmt.Errorf("unable to connect to the content admin app: %w", err)
		return
	}
	defer gConn.Close()

	client := v1.NewGeoContentServiceClient(gConn)
	req := v1.GetContentByIdRequest{Id: int32(cityID)}

	cityImage, err = client.GetCityImage(ctx, &req)
	if err != nil {
		apiErr = fmt.Errorf("content admin returned an error: %w", err)
		return
	}
	return
}
