// Package base provides tools for reading uint32 separated binary data.
package base

import (
	"bytes"
	"crypto/md5"
	"encoding/binary"
	"errors"
	"fmt"
	"hash"
	"io"
	"os"
)

var ErrStopIteration = errors.New("no more entries")

const ErrorPrefix = "travel.library.go.dicts.bytes.iterator"

// BytesIterator provides possibility to iterate over byte sequences of int32 separated binary data
// while returning chunks of these binary data.
type BytesIterator struct {
	reader io.Reader
}

func BuildIteratorFromFile(path string) (*BytesIterator, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, fmt.Errorf("%s:BuildIteratorFromFile: %w", ErrorPrefix, err)
	}

	return &BytesIterator{reader: f}, nil
}

func BuildIteratorFromBytes(b []byte) (*BytesIterator, error) {
	return &BytesIterator{reader: bytes.NewReader(b)}, nil
}

func BuildIteratorFromReader(r io.Reader) (*BytesIterator, error) {
	return &BytesIterator{reader: r}, nil
}

//Every call to Next will return next chunk of data or error
//error == ErrStopIteration means that there is no more chunks
func (it *BytesIterator) Next() ([]byte, error) {
	var structLen int32
	err := binary.Read(it.reader, binary.LittleEndian, &structLen)
	if err != nil {
		if errors.Is(err, io.EOF) {
			return nil, ErrStopIteration
		}
		return nil, fmt.Errorf("%s.BytesIterator:Next: %w", ErrorPrefix, err)
	}
	structBytes := make([]byte, structLen)
	_, err = it.reader.Read(structBytes)
	if err != nil {
		if errors.Is(err, io.EOF) {
			return nil, fmt.Errorf("%s.BytesIterator:Next: unexpected EOF, %w", ErrorPrefix, err)
		}
		return nil, fmt.Errorf("%s.BytesIterator:Next: %w", ErrorPrefix, err)
	}
	return structBytes, nil
}

//Populate will copy all chunks of bytes to some io.Writer implementation
//(and all repositories implements io.Writer interface)
func (it *BytesIterator) Populate(w io.Writer) error {
	for {
		entry, err := it.Next()
		if err == ErrStopIteration {
			return nil
		}
		if err != nil {
			return fmt.Errorf("%s.BytesIterator:Populate: error while iterating %w", ErrorPrefix, err)
		}
		_, err = w.Write(entry)
		if err != nil {
			return fmt.Errorf("%s.BytesIterator:Populate: error while writing %w", ErrorPrefix, err)
		}
	}
}

func (it *BytesIterator) ToChan(entries chan<- []byte, errors chan<- error) {
	defer close(errors)
	defer close(entries)
	for {
		entry, err := it.Next()
		if err == ErrStopIteration {
			errors <- err
			break
		}
		if err != nil {
			errors <- fmt.Errorf("%s.BytesIterator:Populate: error while iterating %w", ErrorPrefix, err)
			break
		}
		entries <- entry
	}
}

// BytesGenerator writes int32 separated binary data with provided writer
type BytesGenerator struct {
	writer io.Writer
	hash   hash.Hash
}

func BuildGeneratorForWriter(w io.Writer) (*BytesGenerator, error) {
	return &BytesGenerator{
		writer: w,
		hash:   md5.New(),
	}, nil
}

func (g *BytesGenerator) Write(data []byte) error {
	structLen := int32(len(data))
	err := binary.Write(g.writer, binary.LittleEndian, &structLen)
	if err != nil {
		return fmt.Errorf("%s:Write: %w", ErrorPrefix, err)
	}
	err = binary.Write(g.hash, binary.LittleEndian, &structLen)
	if err != nil {
		return fmt.Errorf("%s:Write: %w", ErrorPrefix, err)
	}
	_, err = g.writer.Write(data)
	if err != nil {
		return fmt.Errorf("%s:Write: %w", ErrorPrefix, err)
	}
	_, err = g.hash.Write(data)
	if err != nil {
		return fmt.Errorf("%s:Write: %w", ErrorPrefix, err)
	}
	return nil
}

func (g *BytesGenerator) HashSum() []byte {
	return g.hash.Sum(nil)
}
