package base

import (
	"fmt"

	"github.com/golang/protobuf/proto"
)

type ProtoAdder interface {
	AddProto(message proto.Message)
}

type Repository struct {
	code       string
	extensions []ProtoAdder
	newProtoFn func() proto.Message
	predicate  func(proto.Message) bool
}

func NewRepository(code string, newProtoFn func() proto.Message) *Repository {
	return &Repository{
		code:       code,
		extensions: make([]ProtoAdder, 0),
		newProtoFn: newProtoFn,
		predicate: func(message proto.Message) bool {
			return true
		},
	}
}

func (r *Repository) WithPredicate(predicate func(proto.Message) bool) *Repository {
	r.predicate = predicate
	return r
}

func (r *Repository) AddExtensions(extensions ...ProtoAdder) {
	r.extensions = append(r.extensions, extensions...)
}

func (r *Repository) Add(message proto.Message) {
	if !r.predicate(message) {
		return
	}

	for _, ext := range r.extensions {
		ext.AddProto(message)
	}
}

func (r *Repository) Write(b []byte) (int, error) {
	if message, err := r.decode(b); err == nil {
		r.Add(message)
		return len(b), nil
	} else {
		return 0, fmt.Errorf("%s.Write: %w", r.code, err)
	}
}

func (r *Repository) decode(b []byte) (proto.Message, error) {
	message := r.newProtoFn()
	if err := proto.Unmarshal(b, message); err != nil {
		return nil, fmt.Errorf("%s.decode: %w", r.code, err)
	}
	return message, nil

}

func (r *Repository) GetSample() proto.Message {
	return r.newProtoFn()
}
