package dynamicresources

import (
	"fmt"
	"io"
	"path/filepath"

	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Populator interface {
	Populate(writer io.Writer) error
}

type Updatable interface {
	UpdateFromSource(Populator) error
}

type Registry struct {
	updatableByPaths    map[string]Updatable
	path                string
	useDynamicResources bool
}

func NewRegistry(path string, useDynamicResources bool) *Registry {
	return &Registry{
		updatableByPaths:    make(map[string]Updatable),
		path:                path,
		useDynamicResources: useDynamicResources,
	}
}

func (r *Registry) Register(path string, u Updatable) {
	r.updatableByPaths[path] = u
}

func (r *Registry) RegisterBatch(updatableByPaths map[string]Updatable) {
	for path, updatable := range updatableByPaths {
		r.Register(path, updatable)
	}
}

func (r *Registry) OnUpdate() error {
	dir, err := getResourcePath(r.path, r.useDynamicResources)
	if err != nil {
		return fmt.Errorf("unable to handle resource path: %w", err)
	}

	for file, repo := range r.updatableByPaths {
		if err := updateFromResource(filepath.Join(dir, file), repo); err != nil {
			return fmt.Errorf("api.internal.pkg.dicts.Registry.OnUpdate: %w", err)
		}
	}
	return nil
}

func getResourcePath(resourcePath string, useDynamicResources bool) (string, error) {
	if useDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(resourcePath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}

func updateFromResource(path string, updatable Updatable) error {
	bytesIterator, err := base.BuildIteratorFromFile(path)
	if err != nil {
		return fmt.Errorf("unable to build dict iterator by path %s: %w", path, err)
	}
	if err := updatable.UpdateFromSource(bytesIterator); err != nil {
		return fmt.Errorf("unable to update dict by path %s: %w", path, err)
	}
	return nil
}
