package errorbooster

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"github.com/go-resty/resty/v2"
)

type DeployConfig struct {
	Host           string        `config:"ERROR_BOOSTER_HTTP_HOST,optional" yaml:"host"`
	Port           int           `config:"ERROR_BOOSTER_HTTP_PORT,optional" yaml:"port"`
	Datacenter     string        `config:"DEPLOY_NODE_DC,optional"  yaml:"datacenter"`
	ServiceName    string        `yaml:"service-name"`
	RequestTimeout time.Duration `yaml:"request-timeout"`
}

type DeployClient struct {
	endpoint       string
	prefix         string
	env            string
	additionalData map[string]interface{}
	service        string
	httpClient     *resty.Client
}

func NewDeployClient(
	config DeployConfig,
	env string,
) (Client, error) {
	if config.Host == "" || config.Port == 0 {
		return NewNullClient()
	}

	if config.Datacenter == "" {
		config.Datacenter = "none"
	}

	additionalData := map[string]interface{}{
		"datacenter": config.Datacenter,
	}

	return &DeployClient{
		endpoint:       fmt.Sprintf("http://%s:%d/errorbooster", config.Host, config.Port),
		prefix:         prefixFromStringArray([]string{config.ServiceName}),
		env:            env,
		additionalData: additionalData,
		service:        config.ServiceName,
		httpClient:     resty.New().SetTimeout(config.RequestTimeout),
	}, nil
}

func (client *DeployClient) SendError(err error) error {
	msg := Message{
		Message: err.Error(),
		Stack:   fmt.Sprintf("%+v", err),
	}
	return client.SendMessage(msg)
}

func (client *DeployClient) SendMessage(msg Message) error {
	msg = client.prepareMessage(msg)
	var errResponse map[string]interface{}
	response, err := client.httpClient.R().
		SetBody(msg).
		SetError(errResponse).
		Post(client.endpoint)
	if err != nil {
		return err
	} else if !response.IsSuccess() {
		return xerrors.Errorf("Error while executing POST request to error booster: %T", errResponse)
	}

	return nil
}

func (client *DeployClient) prepareMessage(msg Message) Message {
	msg.Project = projectName
	msg.Message = fmt.Sprintf("%s %s", client.prefix, msg.Message)
	msg.Env = client.env
	msg.Service = client.service

	if msg.Timestamp == 0 {
		msg.Timestamp = time.Now().UnixMilli()
	}

	if msg.Additional == nil {
		msg.Additional = client.additionalData
	} else {
		msg.Additional = mergeDicts(client.additionalData, msg.Additional)
	}

	return msg
}
