package errutil

import (
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/x/xruntime"
)

// NewTracedError creates a TracedError instance with passed error.
// If you need ability to generate stack frames, it should be built into passed error via StackTracer interface.
// Basically, you may use xerrors.Errorf() or xerrors.New() from /library/go/core/xerrors package.
func NewTracedError(err error, options ...ErrorOption) *TracedError {
	if err == nil {
		err = xerrors.SkipErrorf(1, "TracedError")
	}
	tracedErr := &TracedError{
		error:  err,
		fields: make(map[string]interface{}),
	}
	for _, option := range options {
		option(tracedErr)
	}
	return tracedErr
}

type ErrorOption func(*TracedError)

func Fields(fields map[string]interface{}) ErrorOption {
	return func(err *TracedError) {
		for k, v := range fields {
			err.fields[k] = v
		}
	}
}

func Field(name string, value interface{}) ErrorOption {
	return Fields(map[string]interface{}{name: value})
}

// TracedError is a type of error, that you can use on its own, or easily create custom error type from it.
// It reveals some interfaces of underlying error and includes ability to add some meta information in Fields.
//
// Try using it like this:
//
//    errutil.NewTracedError(xerrors.Errorf("I am an error"))
//
// Or implement a custom errror type like so
//
//    type customError {*errutil.TracedError}
//    customError{errutil.NewTracedError(xerrors.Errorf("I am a custom error"))}
//
// When implementing custom error it's important to use *errutil.TracedError type, instead of error interface,
// because it allows you not to wrap StackTrace methods or any other useful methods, that might arise
// for this base type in the future
type TracedError struct {
	error
	fields map[string]interface{}
}

func (e *TracedError) StackTrace() *xruntime.StackTrace {
	if e.error == nil {
		return nil
	}
	framer, ok := e.error.(xerrors.ErrorStackTrace)
	if !ok {
		return nil
	}
	return framer.StackTrace()
}

func (e *TracedError) Fields() map[string]interface{} {
	return e.fields
}

func (e *TracedError) Unwrap() error {
	return xerrors.Unwrap(e.error)
}
