package funcnames

import (
	"fmt"
	"reflect"
	"regexp"
	"runtime"
	"strings"
)

var fnName, fullFnName Caller

func init() {
	fnName = BuildName(BuildName)
	fullFnName = BuildName(BuildFullName)
}

// runtime return func name as "{dir0}/{dir2}/{package>}[.{struct_name}].{method_name}[-fm]"
func BuildName(fn interface{}) Caller {
	path := getFunctionPath(fn, fnName)
	parent, fnName := getFuncNameParts(path)
	parent = parent[strings.LastIndex(parent, "/")+1:]
	parent = unwrapPointerName(parent)
	return Caller(fmt.Sprintf("%s.%s", parent, fnName))
}

// runtime return func name as "{dir0}/{dir2}/{package>}[.{struct_name}].{method_name}[-fm]"
func BuildFullName(fn interface{}) Caller {
	path := getFunctionPath(fn, fullFnName)
	parent, fnName := getFuncNameParts(path)

	{
		const arcadiaPrefix = "a.yandex-team.ru/"
		const travelPrefix = "travel/"
		parent = strings.TrimPrefix(parent, arcadiaPrefix)
		parent = strings.TrimPrefix(parent, travelPrefix)
	}

	parent = strings.ReplaceAll(parent, "/", ".")
	parent = unwrapPointerName(parent)
	return Caller(fmt.Sprintf("%s.%s", parent, fnName))
}

func getFunctionPath(fn interface{}, caller Caller) string {
	value := reflect.ValueOf(fn)
	if value.Kind() != reflect.Func {
		caller.Panic("expect function")
	}

	return runtime.FuncForPC(value.Pointer()).Name()
}

func getFuncNameParts(path string) (string, string) {
	path = strings.TrimSuffix(path, "-fm")

	sepIdx := strings.LastIndex(path, ".")
	return path[:sepIdx], path[sepIdx+1:]
}

var pointerNamePattern = regexp.MustCompile(`\(\*([a-zA-Z]*)\)`)

func unwrapPointerName(path string) string {
	return pointerNamePattern.ReplaceAllString(path, "$1")
}
