package geobase

import (
	"errors"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/geobase"
	http2 "a.yandex-team.ru/travel/library/go/geobase/http"
)

type baseInterface interface {
	GetRegionByIP(ip string, crimea ...geobase.CrimeaStatus) (*geobase.Region, error)
	GetCountryID(id geobase.ID, crimea ...geobase.CrimeaStatus) (geobase.ID, error)
	GetRegionByID(id geobase.ID, crimea ...geobase.CrimeaStatus) (*geobase.Region, error)
	GetParentsIDs(id geobase.ID, crimea ...geobase.CrimeaStatus) ([]geobase.ID, error)
	GetLinguistics(id geobase.ID, lang string) (*geobase.Linguistics, error)
	GetTimezoneByID(id geobase.ID) (*geobase.Timezone, error)
	Destroy()
	GetRegionByLocation(latitude float64, longitude float64, status ...geobase.CrimeaStatus) (*geobase.Region, error)
}

type Wrapper struct {
	logger log.Logger
	base   baseInterface
}

func NewGeobase(config Config, logger log.Logger) (Geobase, error) {
	fmt.Printf("geobase run mode %s \n", config.Mode)
	if config.Mode == ModeHTTP {
		return &Wrapper{
			logger: logger,
			base: http2.NewClient(
				&http.Client{Timeout: http2.DefaultTimeout},
				config.DevelopmentAPIHost,
			),
		}, nil
	}
	if config.Mode == ModeStub {
		return StubGeobase{}, nil
	}

	base, err := geobase.New(geobase.WithDatafilePath(config.Path))
	if err != nil {
		if errors.Is(err, geobase.ErrNotSupported) {
			return nil, err
		}
		return nil, fmt.Errorf("unable to construct geobase from file %s: %w", config.Path, err)
	}
	return &Wrapper{
		base:   base,
		logger: logger,
	}, nil
}

func (w *Wrapper) GetLinguistics(geoID int, lang string) (*geobase.Linguistics, error) {
	return w.base.GetLinguistics(geobase.ID(geoID), lang)
}

func (w *Wrapper) GetRegion(geoID int) (*geobase.Region, error) {
	return w.base.GetRegionByID(geobase.ID(geoID))
}
func (w *Wrapper) GetRegionByIP(ip string) (*geobase.Region, error) {
	return w.base.GetRegionByIP(ip)
}

func (w *Wrapper) GetCountryID(geoID int, crimea geobase.CrimeaStatus) (int, error) {
	id, err := w.base.GetCountryID(geobase.ID(geoID), crimea)
	if err != nil {
		return 0, fmt.Errorf("failed to get country id %d", geoID)
	}
	return int(id), nil
}

func (w *Wrapper) RoundToRegion(geoID int, regionType geobase.RegionType) (*geobase.Region, error) {
	parentIDs, err := w.base.GetParentsIDs(geobase.ID(geoID))
	if err != nil {
		return nil, fmt.Errorf("failed to get geo region %d parents", geoID)
	}
	for _, id := range parentIDs {
		parent, _ := w.base.GetRegionByID(id)
		if parent.Type == regionType {
			return parent, nil
		}
	}
	w.logger.Warn("no parent for region", log.Any("regionType", regionType), log.Int("geoID", geoID))
	return nil, fmt.Errorf("no parent of type %v for region %d", regionType, geoID)
}

func (w *Wrapper) GetLocationByID(geoID int) (*time.Location, error) {
	const funcName = "geobase.GeobaseWrapper.GetLocationByID"

	region, err := w.base.GetRegionByID(geobase.ID(geoID))
	if err != nil {
		return nil, fmt.Errorf("%s: unable to find region: %w", funcName, err)
	}

	w.logger.Debug("found region in geobase",
		log.String("name", region.Name),
		log.String("timezoneName", region.TimezoneName),
	)

	loc, err := time.LoadLocation(region.TimezoneName)
	if err != nil {
		return nil, fmt.Errorf("%s: unable to load location: %w", funcName, err)
	}

	return loc, nil
}

func (w *Wrapper) GetRegionByLocation(latitude float64, longitude float64) (*geobase.Region, error) {
	return w.base.GetRegionByLocation(latitude, longitude)
}

func (w Wrapper) Destroy() {
	w.base.Destroy()
}
