package http

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"path"
	"strconv"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/yandex/geobase"
	models2 "a.yandex-team.ru/travel/library/go/geobase/http/models"
)

type Client struct {
	httpClient          *http.Client
	apiHost             string
	backOffPolicyGetter func() backoff.BackOff
	requestTimeout      time.Duration
}

// https://wiki.yandex-team.ru/http-geobase/api-v1
func NewClient(httpClient *http.Client, apiHost string, opts ...Option) *Client {
	client := &Client{httpClient: httpClient, apiHost: apiHost}

	defaultOpts := []Option{withDefaultBackOffPolicy, withDefaultRequestTimeout}
	for _, opt := range append(defaultOpts, opts...) {
		opt(client)
	}
	return client
}

func (c *Client) GetLinguistics(id geobase.ID, lang string) (*geobase.Linguistics, error) {
	request := c.buildRequest("linguistics_for_region", map[string]string{"id": strconv.Itoa(int(id)), "lang": lang})
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	model := models2.Linguistics{}
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return nil, err
	}
	result := geobase.Linguistics(model)
	return &result, nil
}

func (c *Client) GetTimezoneByID(id geobase.ID) (*geobase.Timezone, error) {
	request := c.buildRequest("tzinfo", map[string]string{"id": strconv.Itoa(int(id))})
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	model := models2.Timezone{}
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return nil, err
	}
	return &geobase.Timezone{
		Name:   model.Name,
		Abbr:   model.Abbr,
		Dst:    model.Dst,
		Offset: time.Duration(model.Offset) * time.Second,
	}, nil
}

func (c *Client) GetRegionByID(id geobase.ID, crimea ...geobase.CrimeaStatus) (*geobase.Region, error) {
	params := map[string]string{"id": strconv.Itoa(int(id))}
	c.addCrimea(crimea, params)
	request := c.buildRequest("region_by_id", params)
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	model := models2.Region{}
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return nil, err
	}
	return &geobase.Region{
		ID:           model.ID,
		ParentID:     model.ParentID,
		CapitalID:    model.CapitalID,
		CityID:       model.CityID,
		Type:         model.Type,
		Name:         model.Name,
		TimezoneName: model.TimezoneName,
	}, nil
}

func (c *Client) GetRegionByIP(ip string, crimea ...geobase.CrimeaStatus) (*geobase.Region, error) {
	params := map[string]string{"ip": ip}
	c.addCrimea(crimea, params)
	request := c.buildRequest("region_by_ip", params)
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	model := models2.Region{}
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return nil, err
	}
	return &geobase.Region{
		ID:           model.ID,
		ParentID:     model.ParentID,
		CapitalID:    model.CapitalID,
		CityID:       model.CityID,
		Type:         model.Type,
		Name:         model.Name,
		TimezoneName: model.TimezoneName,
	}, nil
}

func (c *Client) GetCountryID(id geobase.ID, crimea ...geobase.CrimeaStatus) (geobase.ID, error) {
	params := map[string]string{"id": strconv.Itoa(int(id))}
	c.addCrimea(crimea, params)
	request := c.buildRequest("find_country", params)
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return 0, err
	}
	var model int
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return 0, err
	}
	return geobase.ID(model), nil
}

func (c *Client) GetRegionByLocation(latitude float64, longitude float64, crimea ...geobase.CrimeaStatus) (*geobase.Region, error) {
	params := map[string]string{"lat": fmt.Sprintf("%f", latitude), "lon": fmt.Sprintf("%f", longitude)}
	c.addCrimea(crimea, params)
	request := c.buildRequest("region_id_by_location", params)
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	var regionID geobase.ID = 0
	if err := json.Unmarshal(responseBytes, &regionID); err != nil {
		return nil, err
	}
	return c.GetRegionByID(regionID)
}

func (c *Client) GetParentsIDs(id geobase.ID, crimea ...geobase.CrimeaStatus) ([]geobase.ID, error) {
	params := map[string]string{"id": strconv.Itoa(int(id))}
	c.addCrimea(crimea, params)
	request := c.buildRequest("parents", params)
	responseBytes, err := c.doRequest(request)
	if err != nil {
		return nil, err
	}
	var model []int
	if err := json.Unmarshal(responseBytes, &model); err != nil {
		return nil, err
	}
	parentIDs := make([]geobase.ID, 0, len(model))
	for _, parentID := range model {
		parentIDs = append(parentIDs, geobase.ID(parentID))
	}
	return parentIDs, nil
}

func (c *Client) addCrimea(crimea []geobase.CrimeaStatus, params map[string]string) {
	if len(crimea) > 0 {
		if crimea[0] == geobase.CrimeaInRU {
			params["crimea_status"] = "ru"
		} else {
			params["crimea_status"] = "ua"
		}
	}
}

func (c *Client) Destroy() {
	c.httpClient.CloseIdleConnections()
}

func (c *Client) buildRequest(apiMethod string, params map[string]string) *http.Request {
	requestURL, _ := url.Parse(c.apiHost)
	requestURL.Path = path.Join(requestURL.Path, apiMethod)
	query := url.Values{}
	for key, value := range params {
		query[key] = []string{value}
	}
	requestURL.RawQuery = query.Encode()
	request, _ := http.NewRequest(http.MethodGet, requestURL.String(), nil)
	return request
}

func (c *Client) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}

func (c *Client) doRequest(request *http.Request) ([]byte, error) {
	var byteResponse []byte
	requestFunc := func() error {
		response, err := c.httpClient.Do(request)
		if err != nil {
			return err
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return err
		}
		if response.StatusCode == http.StatusNotFound {
			return backoff.Permanent(err)
		}
		if response.StatusCode != http.StatusOK {
			err := fmt.Errorf("http-geobase responded"+
				" with code %d: %s", response.StatusCode, string(byteResponse))
			if response.StatusCode == http.StatusBadRequest {
				return backoff.Permanent(err)
			}
			return err
		}
		return nil
	}
	if err := c.retryRequest(requestFunc); err != nil {
		return nil, err
	}
	return byteResponse, nil
}
